<?php

namespace SEOAIC\repositories;

use SEOAIC\repositories\PostRepository;
use SEOAIC\SEOAIC_POSTS;

class PostRevisionRepository extends PostRepository
{
    public function __construct()
    {
    }

    public function hasRevision(int $postId): bool
    {
        $posts = $this->getByPostID($postId);

        return !empty($posts);
    }

    /**
     * Gets a list of revisions for provided post ID
     * @param int $postId
     * @return WP_Post[] Array of WP_Post records ordered by date DESC
     */
    public function getByPostID(int $postId): array
    {
        $args = [
            'post_parent'   => $postId,
            'post_type'     => SEOAIC_POSTS::REVISION_POST_TYPE,
            'post_status'   => SEOAIC_POSTS::REVISION_STATUS,
            'numberposts'   => -1,
            'lang'          => '',
            'orderby'       => 'date',
            'order'         => 'DESC',
        ];

        $posts = get_posts($args);

        return $posts;
    }

    /**
     * Get all revisions
     * return array WP_Post
     */
    public function getAll()
    {
        $args = [
            'post_type'     => SEOAIC_POSTS::REVISION_POST_TYPE,
            'post_status'   => SEOAIC_POSTS::REVISION_STATUS,
            'numberposts'   => -1,
            'lang'          => '',
        ];

        $posts = get_posts($args);

        return $posts;
    }


    /**
     * Creates a post's revision
     * @param array $postData Post's data to be saved as revision. ID is required
     * @return int
     */
    public function create(array $postData): int
    {
        if (empty($postData['ID'])) {
            return 0;
        }

        $postsFound = $this->getByIDs([$postData['ID']]);

        if (
            empty($postsFound)
            || empty($postsFound[0])
            || !is_a($postsFound[0], 'WP_Post')
        ) {
            return 0;
        }

        $post = $postsFound[0];

        return parent::create([
            'post_type'     => SEOAIC_POSTS::REVISION_POST_TYPE,
            'post_status'   => SEOAIC_POSTS::REVISION_STATUS,
            'post_parent'   => $post->ID,
            'post_title'    => $post->post_title,
            'post_content'  => $post->post_content,
        ]);
    }

    /**
     * Remove revision by its ID
     * @param int $id
     */
    public function delete(int $id)
    {
        wp_delete_post($id);
    }

    public function restore(int $postId): bool
    {
        $posts = $this->getByIDs([$postId]);

        if (
            empty($posts)
            || empty($posts[0])
        ) {
            return false;
        }

        $post = $posts[0];
        $postRevisions = $this->getByPostID($postId);
        $lastPostRevision = !empty($postRevisions[0]) ? $postRevisions[0] : null;

        if (empty($lastPostRevision)) {
            return false;
        }

        $restoreRes = wp_update_post([
            'ID'            => $post->ID,
            'post_content'  => $lastPostRevision->post_content,
        ]);

        if ($restoreRes) {
            // $this->delete($lastPostRevision->ID); // delete last

            foreach ($postRevisions as $postRevision) { // delete all
                $this->delete($postRevision->ID);
            }
        }

        return (bool) $restoreRes;
    }
}
