<?php

namespace SEOAIC\helpers;

use Exception;

class ImageHelpers
{
/**
     * Upload a file to the media library using a URL.
     *
     * @param string $url URL to be uploaded
     * @param array $attachmentData Array of values that can be used for attachment, e.g. post_title
     * @param array $metaData Array of values that can be used as meta fields, e.g. _wp_attachment_image_alt
     *
     * @return array|false Returns array with attachment ID and array of urls for media sizes ('thumbnail','medium','large','full' )
     */
    public static function uploadFromURL(string $url, array $attachmentData = [], array $metaData = [])
    {
        try {
            // $filename = pathinfo($url, PATHINFO_FILENAME);
            $extension = pathinfo($url, PATHINFO_EXTENSION);

            if (
                !self::isAllowedExtension($extension)
                || !self::isAllowedContentType($url)
            ) {
                return false;
            }

            $wpUploadDir = wp_upload_dir();
            $upload_path = $wpUploadDir['path'];
            $origFilename = basename($url);
            $localFile = $upload_path . '/' . $origFilename;

            $response = wp_safe_remote_get($url, [
                'timeout'   => 80,
                'sslverify' => seoaic_ssl_verifypeer(),
                'headers'   => [],
            ]);

            if (is_wp_error($response)) {
                // error_log('$response '.print_r($response, true));
                throw new Exception('Image fetch failed.');
            }

            $body = wp_remote_retrieve_body($response);

            file_put_contents($localFile, $body);

            // An extension is required or else WordPress will reject the upload
            if (!$extension) {
                // Look up mime type, example: "/photo.png" -> "image/png"
                $mime = mime_content_type($body);
                $mime = is_string($mime) ? sanitize_mime_type($mime) : false;

                // Only allow certain mime types because mime types do not always end in a valid extension
                $mimeExtensions = [
                    // mime_type    => extension (no period)
                    'image/jpg'     => 'jpg',
                    'image/jpeg'    => 'jpeg',
                    'image/gif'     => 'gif',
                    'image/png'     => 'png',
                    'image/webp'    => 'webp',
                ];

                if (isset($mimeExtensions[$mime])) {
                    // Use the mapped extension
                    $extension = $mimeExtensions[$mime];
                } else {
                    // Could not identify extension. Clear temp file and abort.
                    // wp_delete_file($tmp);
                    throw new Exception('Could not identify extension');
                }
            }

            $readableFilename = !empty($attachmentData['readable_filename']) ? $attachmentData['readable_filename'] : $origFilename;
            $extInFilename = false !== strpos($readableFilename, '.' . $extension);
            $args = [
                'name' => $extInFilename ? $readableFilename : "$readableFilename.$extension",
                'tmp_name' => $localFile,
            ];

            // Do the upload
            $attachmentID = media_handle_sideload($args, 0, null, [
                'post_title' => $attachmentData['post_title'] ?? '',
            ]);

            if (is_wp_error($attachmentID)) {
                throw new Exception('WP error: ' . $attachmentID->get_error_message());
            }

            if (!empty($metaData)) {
                foreach ($metaData as $key => $value) {
                    if (empty($value)) {
                        continue;
                    }

                    switch ($key) {
                        case 'alt':
                        case '_wp_attachment_image_alt':
                            update_post_meta($attachmentID, '_wp_attachment_image_alt', $value);
                            break;
                    }
                }
            }

            return [
                'id'    => $attachmentID,
                'sizes' => [
                    'thumbnail' => wp_get_attachment_image_url($attachmentID, 'thumbnail'),
                    'medium'    => wp_get_attachment_image_url($attachmentID, 'medium'),
                    'large'     => wp_get_attachment_image_url($attachmentID, 'large'),
                    'full'      => wp_get_attachment_image_url($attachmentID, 'full'),
                ],
            ];

        } catch (Exception $e) {
            error_log('[SEOAIC][ERROR]: '.print_r($e->getMessage(), true));
            return false;
        }

        return false;
    }

    public static function isAllowedExtension($extension): bool
    {
        if (empty($extension)) {
            return true;
        }

        $allowedExtensions = [
            "png",
            "apng",
            "avif",
            "gif",
            "jpg",
            "jpeg",
            "jfif",
            "pjpeg",
            "pjp",
            "svg",
            "webp",
        ];

        if (!in_array($extension, $allowedExtensions)) {
            return false;
        }

        return true;
    }

    public static function isAllowedContentType(string $url)
    {
        if (empty($url)) {
            return true;
        }

        try {
            $response = wp_get_http_headers($url);

            if (!$response) {
                return true;
            }

            $headers = $response->getAll();
            $contentType = $headers['content-type'];
            list($type, $ext) = explode('/', $contentType);

            if (
                "image" != $type
                || !self::isAllowedExtension($ext)
            ) {
                return false;
            }
        } catch (Exception $e) {
        }

        return true;
    }
}
