<?php

namespace SEOAIC\content_editor_blocks\providers;

use SEOAIC\interfaces\BlocksInterface;
use SEOAIC\content_editor_blocks\blocks\FAQBlock;

class SEOAICBlocks implements BlocksInterface
{
    private $blockPrefix = 'seoaic';
    private $schema = '';

    public function faq($questions=[])
    {
        if (
            empty($questions)
            || !is_array($questions)
        ) {
            return '';
        }

        array_walk($questions, function(&$item, $i) {
            $item = [
                'id' => $i,
                'title' => $item['question'],
                'content' => $item['answer'],
            ];
        });

        $attrs = [
            'questions' => $questions,
        ];

        $block = new FAQBlock(
            $this->blockPrefix,
            $attrs,
            $this->faqHTML($attrs)
        );

        $html = $block->render();

        return $html;
    }

    private function faqHTML($attributes)
    {
        $paramsJSON = htmlentities(json_encode($attributes['questions'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES), ENT_NOQUOTES | ENT_SUBSTITUTE);

        return "[seoaic_faq]" . $paramsJSON . "[/seoaic_faq]";
    }

    /**
     * Makes script for schema markup
     * @return string
     */
    public function getSchemaScript()
    {
        return '<script type="application/ld+json">' . $this->schema . '</script>';
    }

    public function getSchema()
    {
        return $this->schema;
    }

    /**
     * Transforms array of FAQ questions/answers into schema markup
     * @param array $questions
     * @return object SEOAICBlocks instance
     */
    public function doFAQSchema($questions)
    {
        $data = [
            '@context' => 'https://schema.org',
            '@graph'   => $this->addGraph($questions),
        ];
        $schema = wp_json_encode(wp_kses_post_deep($data), JSON_UNESCAPED_SLASHES);
        $this->schema = $schema;

        return $this;
    }

    public function addGraph($questions=[])
    {
        if (
            empty($questions)
            || !is_array($questions)
        ) {
            return '';
        }

        $graph = [
            '@type'      => 'FAQPage',
            'mainEntity' => [],
        ];

        $permalink = get_permalink() . '#';
        foreach ($questions as $question) {
            if (
                empty($question['title'])
                || empty($question['content'])
            ) {
                continue;
            }

            $id = 'seoaic-faq-' . md5($question['title']);
            $graph['mainEntity'][] = [
                '@type'          => 'Question',
                'url'            => esc_url($permalink . $id),
                'name'           => wp_strip_all_tags($question['title']),
                'acceptedAnswer' => [
                    '@type' => 'Answer',
                    'text'  => strip_tags($question['content'], '<h1><h2><h3><h4><h5><h6><br><ol><ul><li><a><p><b><strong><i><em>'),
                ],
            ];
        }

        return $graph;
    }
}
