/**
 * Registers a new block provided a unique name and an object defining its behavior.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-registration/
 */

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * All files containing `style` keyword are bundled together. The code used
 * gets applied both to the front of your site and to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './style.scss';

/**
 * Internal dependencies
 */
import Edit from './edit';
import metadata from './block.json';
import { addFilter } from '@wordpress/hooks';
import { registerPlugin } from '@wordpress/plugins';
import { registerBlockType } from '@wordpress/blocks';
import { select, useSelect } from '@wordpress/data';
import { createHigherOrderComponent } from '@wordpress/compose';
import { Fragment } from '@wordpress/element';
import { PluginDocumentSettingPanel } from '@wordpress/editor';
import store from './store/global';
import ControlsText from './controls-text';
import ControlsImage from './controls-image';
import InProgressPlaceholder from './components/in-progress-placeholder';
import SEOAICSidebar from './components/sidebar/sidebar';
import ThumbnailGenerator from './components/thumbnail/thumbnail-generator';


/**
 * Every block starts by registering a new block type definition.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-registration/
 */
registerBlockType( metadata.name, {
    edit: Edit,
} );


// const { Fragment } = wp.element;

const textBlocks = ['core/paragraph', 'core/heading', 'core/list']; // Enable control to some blocks
const imageBlocks = ['core/image'];

/**
 * Add Custom Block Controls
 */
const addBlockControls = createHigherOrderComponent((BlockEdit) => {
    return (props) => {
        const { name, isSelected, clientId } = props;

        if (textBlocks.includes(name)) {
            // const inProgress = useSelect( (select) => {
            //     return select('SEOAIBlocks').getBlockLoading(clientId);
            // });

            // if (inProgress) {
            //     return (
            //         <InProgressPlaceholder/>
            //     );
            // } else {
                return (
                    <Fragment>
                        { isSelected && <ControlsText
                            {...props}
                        /> }
                        <BlockEdit {...props} />
                    </Fragment>
                );
            // }

        } else if (imageBlocks.includes(name)) {
            return (
                <Fragment>
                    { isSelected && <ControlsImage
                        {...props}
                    /> }
                    <BlockEdit { ...props } />
                </Fragment>
            );

        } else {
            return <BlockEdit {...props} />;
        }
    };
}, 'addBlockControls');

addFilter(
    'editor.BlockEdit',
    'seoaic/add-block-controls',
    addBlockControls
);

// Sidebar
(function(wp, React) {
    registerPlugin('seoaic-gutenberg-sidebar-plugin', {
        render: SEOAICSidebar,
    });
})(window.wp, window.React);

// add custom image attributes
addFilter(
    'blocks.getBlockAttributes',
    'seoaic/image-attributes',
    (blockAttributes, blockType, innerHTML, attributes) => {
        if ('core/image' === blockType.name) {
            blockType.attributes.userPrompt = {
                type: 'string',
                default: ''
            };
        }

        return blockAttributes;
    }
);


const PluginPostSettingPanelThumbnailGenerator = () => {
    const canUse = wpApiSettings.sidebar_slots_available && "1" == wpApiSettings.sidebar_slots_available;

    if (canUse) { // fix to prevent errors in old WP versions
        return (
            <PluginDocumentSettingPanel
                name="thumbnail-generator-panel"
                title="SEOAI Featured Image Generator"
                className="seoaic-thumbnail-generator-panel"
                icon="screenoptions seoaic-icon"
            >
                <ThumbnailGenerator />
            </PluginDocumentSettingPanel>
        );
    } else {
        return ('');
    }
};

registerPlugin('seoaic-post-thumbnail', {
    render: PluginPostSettingPanelThumbnailGenerator
});