import { __ } from '@wordpress/i18n';
import { useState, useEffect } from 'react';
import Message from './answer-types/message';
import MultiChoice from './answer-types/multi-choice';
import OpenTextField from './answer-types/open-text-field';
import SingleChoice from './answer-types/single-choice';
import ContactFields from './answer-types/contact-fields';
import arrayMove from '../utils/arrayMove';

const AnswerComponent = ({
    selectedOptionType,
    steps,
    currentStep,
    currentIndex,
    setStepField
}) => {
    const [ CTAStepsOptions, setCTAStepsOptions ] = useState([]);

    const makeCTAStepsOptions = () => {
        const options = [{
            value: '',
            label: __("Select the Step", "seoaic")
        }];

        steps.forEach((step, i) => {
            if (currentIndex != i) {
                options.push({
                    value: step.id,
                    label: "#" + (i + 1) + ': ' + step.question
                });
            }
        });

        options.push({
            label: __('- Finish -', 'seoaic'),
            value: '#finish#',
        });

        return options;
    };

    const addFormItem = (option) => {
        const currentStepFormItems = [...currentStep.formItems, option];

        setStepField(currentStep.id, 'formItems', currentStepFormItems);
    };

    const setFormItems = (options) => {
        setStepField(currentStep.id, 'formItems', options);
    };

    const updateFormItem = (key, field, newValue) => {
        const currentStepFormItems = currentStep.formItems.map((item, i) => {
            if (key == i) {
                item[field] = newValue;
            }

            return item;
        });

        setStepField(currentStep.id, 'formItems', currentStepFormItems);
    };

    const removeFormItem = (key) => {
        const currentStepFormItems = currentStep.formItems.filter((value, i) => {
            return key != i;
        });

        setStepField(currentStep.id, 'formItems', currentStepFormItems);
    };

    const addButton = (button) => {
        const currentStepButtons = [...currentStep.buttons, button];

        setStepField(currentStep.id, 'buttons', currentStepButtons);
    };

    const updateButton = (key, field, newValue) => {
        const currentStepButtons = currentStep.buttons.map((item, i) => {
            if (key == i) {
                item[field] = newValue;
            }

            return item;
        });

        setStepField(currentStep.id, 'buttons', currentStepButtons);
    };

    const removeButton = (key) => {
        const currentStepButtons = currentStep.buttons.filter((value, i) => {
            return key != i;
        });

        setStepField(currentStep.id, 'buttons', currentStepButtons);
    };

    const moveItemUp = (itemKey, index) => {
        if (
            currentStep[itemKey].length > 1
            && index > 0
        ) {
            const items = [...currentStep[itemKey]];

            arrayMove(items, index, index - 1);
            setStepField(currentStep.id, itemKey, items);
        }
    };

    const moveItemDown = (itemKey, index) => {
        if (
            currentStep[itemKey].length > 1
            && index + 1 <= currentStep[itemKey].length
        ) {
            const items = [...currentStep[itemKey]];

            arrayMove(items, index, index + 1);
            setStepField(currentStep.id, itemKey, items);
        }
    };

    const updateAutostep = (newValue) => {
        setStepField(currentStep.id, 'autostep', newValue);
    };

    useEffect(() => {
        setCTAStepsOptions(makeCTAStepsOptions());
    }, [steps]);

    if ('single' == selectedOptionType) {
        return (
            <SingleChoice
                steps={ steps }
                currentStep={ currentStep }
                CTAStepsOptions={ CTAStepsOptions }
                addButton={ addButton }
                updateButton={ updateButton }
                removeButton={ removeButton }
                moveItemUp={ moveItemUp }
                moveItemDown={ moveItemDown }
            />
        );

    } else if ('multi' == selectedOptionType) {
        return (
            <MultiChoice
                steps={ steps }
                currentStep={ currentStep }
                CTAStepsOptions={ CTAStepsOptions }
                addFormItem={ addFormItem }
                setFormItems={ setFormItems }
                updateFormItem={ updateFormItem }
                removeFormItem={ removeFormItem }
                addButton={ addButton }
                updateButton={ updateButton }
                moveItemUp={ moveItemUp }
                moveItemDown={ moveItemDown }
            />
        );

    } else if ('textfield' == selectedOptionType) {
        return (
            <OpenTextField
                steps={ steps }
                currentStep={ currentStep }
                CTAStepsOptions={ CTAStepsOptions }
                setFormItems={ setFormItems }
                updateFormItem={ updateFormItem }
                addButton={ addButton }
                updateButton={ updateButton }
            />
        );

    } else if ('text' == selectedOptionType) {
        return (
            <Message
                steps={ steps }
                currentStep={ currentStep }
                CTAStepsOptions={ CTAStepsOptions }
                setFormItems={ setFormItems }
                updateFormItem={ updateFormItem }
                updateAutostep={ updateAutostep }
            />
        );

    } else if ('contact_fields' == selectedOptionType) {
        return (
            <ContactFields
                steps={ steps }
                currentStep={ currentStep }
                CTAStepsOptions={ CTAStepsOptions }
                setFormItems={ setFormItems }
                addFormItem={ addFormItem }
                updateFormItem={ updateFormItem }
                removeFormItem={ removeFormItem }
                addButton={ addButton }
                updateButton={ updateButton }
                moveItemUp={ moveItemUp }
                moveItemDown={ moveItemDown }
            />
        );
    }
};

export default AnswerComponent;