<?php

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\SiteDuplication;

use Inpsyde\MultilingualPress\Attachment;
use Inpsyde\MultilingualPress\Core\Admin\SiteSettingsUpdater;
use Inpsyde\MultilingualPress\Framework\Api\ContentRelations;
use Inpsyde\MultilingualPress\Framework\BasePathAdapter;
use Inpsyde\MultilingualPress\Framework\Factory\NonceFactory;
use Inpsyde\MultilingualPress\Framework\Database\TableDuplicator;
use Inpsyde\MultilingualPress\Framework\Database\TableList;
use Inpsyde\MultilingualPress\Framework\Database\TableReplacer;
use Inpsyde\MultilingualPress\Framework\Database\TableStringReplacer;
use Inpsyde\MultilingualPress\Framework\Filesystem;
use Inpsyde\MultilingualPress\Framework\Http\ServerRequest;
use Inpsyde\MultilingualPress\Framework\Message\MessageFactoryInterface;
use Inpsyde\MultilingualPress\Framework\Module\ModuleManager;
use Inpsyde\MultilingualPress\Framework\Nonce\Context;
use Inpsyde\MultilingualPress\Framework\Nonce\Nonce;
use Inpsyde\MultilingualPress\Framework\Nonce\SiteAwareNonce;
use Inpsyde\MultilingualPress\Framework\Service\Exception\LateAccessToNotSharedService;
use Inpsyde\MultilingualPress\Framework\Service\Exception\NameNotFound;
use Inpsyde\MultilingualPress\Framework\Service\Exception\NameOverwriteNotAllowed;
use Inpsyde\MultilingualPress\Framework\Service\Exception\WriteAccessOnLockedContainer;
use Inpsyde\MultilingualPress\Framework\Setting\Site\SiteSettingMultiView;
use Inpsyde\MultilingualPress\Framework\Setting\Site\SiteSettingsSectionView;
use Inpsyde\MultilingualPress\Core\Admin\NewSiteSettings;
use Inpsyde\MultilingualPress\Framework\Service\BootstrappableServiceProvider;
use Inpsyde\MultilingualPress\Framework\Service\Container;
use Inpsyde\MultilingualPress\Schedule\AjaxScheduleHandler;
use Inpsyde\MultilingualPress\Schedule\Scheduler;
use Inpsyde\MultilingualPress\Schedule\Action\ScheduleActionRequestHandler;
use Inpsyde\MultilingualPress\Schedule\Action\RemoveActionTask;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\AttachmentDuplicatorHandler;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\AttachmentDuplicatorScheduler;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\MaybeScheduleAttachmentDuplication;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\NewSiteScheduleTemplate;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\ScheduleAssetManager;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\SiteScheduleOption;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\RemoveAttachmentIdsTask;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\ScheduleActionsNames;
use Throwable;
use WP_Site;
use Inpsyde\MultilingualPress\SiteDuplication\Settings\ActivatePluginsSetting;
use Inpsyde\MultilingualPress\SiteDuplication\Settings\BasedOnSiteSetting;
use Inpsyde\MultilingualPress\SiteDuplication\Settings\ConnectContentSetting;
use Inpsyde\MultilingualPress\SiteDuplication\Settings\CopyAttachmentsSetting;
use Inpsyde\MultilingualPress\SiteDuplication\Settings\CopyUsersSetting;
use Inpsyde\MultilingualPress\SiteDuplication\Settings\SearchEngineVisibilitySetting;
use Inpsyde\MultilingualPress\SiteDuplication\Settings\ConnectCommentsSetting;
use Inpsyde\MultilingualPress\Core\ServiceProvider as CoreModule;

use function Inpsyde\MultilingualPress\wpHookProxy;
use function Inpsyde\MultilingualPress\wpVersion;

class ServiceProvider implements BootstrappableServiceProvider
{
    public const SITE_DUPLICATION_SUCCESS_ACTIONS_MESSAGES = 'siteDuplication.successActionsMessages';
    public const SCHEDULE_ACTION_ATTACHMENTS_REMOVER_SERVICE = 'siteDuplication.scheduleActionAttachmentsRemover';
    public const SCHEDULE_ACTION_ATTACHMENT_HANDLER_SERVICE = 'siteDuplication.scheduleActionAttachmentHandler';
    public const SITE_DUPLICATION_ACTIONS = 'siteDuplication.actionsService';

    public const FILTER_SUCCESS_ACTIONS_MESSAGES = 'multilingualpress.filter_success_actions_messages';
    public const FILTER_SITE_DUPLICATION_ACTIONS = 'multilingualpress.site_duplication_actions';

    public const SCHEDULE_ACTION_ATTACHMENTS_AJAX_HOOK_NAME = 'multilingualpress_site_duplicator_attachments_schedule_action';

    protected const SCHEDULE_ACTION_ATTACHMENTS_USER_REQUIRED_CAPABILITY = 'create_sites';
    protected const SCHEDULE_ACTION_ATTACHMENTS_NONCE_KEY = 'multilingualpress_attachment_duplicator_action';

    public const SITE_DUPLICATION_FILTER_MLP_TABLES = 'siteDuplication.filterMlpTables';
    public const CONFIGURATION_SITE_DUPLICATION_EXCLUDED_OPTIONS = 'siteDuplication.excludedOptions';
    public const SITE_DUPLICATION_FILTER_EXCLUDED_OPTIONS = 'multilingualpress.site_duplicator.filter_excluded_options';

    /**
     * @inheritdoc
     *
     * phpcs:disable Inpsyde.CodeQuality.FunctionLength.TooLong
     * @param Container $container
     * @throws NameOverwriteNotAllowed
     * @throws WriteAccessOnLockedContainer
     */
    public function register(Container $container)
    {
        // phpcs:enable

        $container->addService(
            ActivePlugins::class,
            static function (): ActivePlugins {
                return new ActivePlugins();
            }
        );

        $container->addService(
            Attachment\Duplicator::class,
            static function (Container $container): Attachment\Duplicator {
                return new Attachment\Duplicator(
                    $container[BasePathAdapter::class],
                    $container[Filesystem::class]
                );
            }
        );

        $container->addService(
            ConnectContentSetting::class,
            static function (): ConnectContentSetting {
                return new ConnectContentSetting();
            }
        );

        $container->addService(
            ActivatePluginsSetting::class,
            static function (): ActivatePluginsSetting {
                return new ActivatePluginsSetting();
            }
        );

        $container->addService(
            BasedOnSiteSetting::class,
            function (Container $container): BasedOnSiteSetting {
                return new BasedOnSiteSetting(
                    $container[\wpdb::class],
                    $this->duplicateNonce($container)
                );
            }
        );

        $container->addService(
            CopyAttachmentsSetting::class,
            static function (): CopyAttachmentsSetting {
                return new CopyAttachmentsSetting();
            }
        );

        $container->addService(
            CopyUsersSetting::class,
            static function (): CopyUsersSetting {
                return new CopyUsersSetting();
            }
        );

        $container->addService(
            ConnectCommentsSetting::class,
            static function (): ConnectCommentsSetting {
                return new ConnectCommentsSetting('mlp-copy-comments');
            }
        );

        $container->addService(
            SearchEngineVisibilitySetting::class,
            static function (): SearchEngineVisibilitySetting {
                return new SearchEngineVisibilitySetting();
            }
        );

        $container->share(
            self::CONFIGURATION_SITE_DUPLICATION_EXCLUDED_OPTIONS,
            static function (): array {
                /**
                 * Remove Jetpack and WordPress VIP-related options.
                 *
                 * @see https://docs.wpvip.com/databases/data-sync/#protected-options
                 */
                return (array) apply_filters(
                    self::SITE_DUPLICATION_FILTER_EXCLUDED_OPTIONS,
                    [
                        'jetpack_options',
                        'jetpack_private_options',
                        'vip_jetpack_connection_pilot_heartbeat',
                        'wordpress_api_key',
                        'vip_search_index_versions',
                        'vip_search_global_index_versions',
                    ]
                );
            }
        );

        $container->addService(
            SiteDuplicator::class,
            function (Container $container): SiteDuplicator {
                return new SiteDuplicator(
                    $container[\wpdb::class],
                    $container[TableList::class],
                    $container[TableDuplicator::class],
                    $container[TableReplacer::class],
                    $container[ActivePlugins::class],
                    $container[ContentRelations::class],
                    $container[ServerRequest::class],
                    $this->duplicateNonce($container),
                    $container->get(ModuleManager::class),
                    $container->get(self::CONFIGURATION_SITE_DUPLICATION_EXCLUDED_OPTIONS)
                );
            }
        );

        $container->addService(
            SiteScheduleOption::class,
            static function (): SiteScheduleOption {
                return new SiteScheduleOption();
            }
        );

        $container->addService(
            AttachmentDuplicatorScheduler::class,
            static function (Container $container): AttachmentDuplicatorScheduler {
                return new AttachmentDuplicatorScheduler(
                    $container[SiteScheduleOption::class],
                    $container[Attachment\Collection::class],
                    $container[Scheduler::class]
                );
            }
        );

        $container->addService(
            Attachment\DatabaseDataReplacer::class,
            static function (Container $container): Attachment\DatabaseDataReplacer {
                return new Attachment\DatabaseDataReplacer(
                    $container[\wpdb::class],
                    $container[TableStringReplacer::class],
                    $container[BasePathAdapter::class]
                );
            }
        );

        $container->addService(
            AttachmentDuplicatorHandler::class,
            static function (Container $container): AttachmentDuplicatorHandler {
                return new AttachmentDuplicatorHandler(
                    $container[SiteScheduleOption::class],
                    $container[Attachment\Duplicator::class],
                    $container[Attachment\Collection::class],
                    $container[Scheduler::class],
                    $container[Attachment\DatabaseDataReplacer::class]
                );
            }
        );

        $container->share(
            ScheduleAssetManager::class,
            static function (Container $container): ScheduleAssetManager {
                return new ScheduleAssetManager(
                    $container->get(SiteScheduleOption::class),
                    $container->get(AjaxScheduleHandler::class),
                    $container->get(NonceFactory::class)->create([self::SCHEDULE_ACTION_ATTACHMENTS_NONCE_KEY]),
                    $container->get(CoreModule::CONFIGURATION_NAME_FOR_URL_TO_CORE_MODULE_ASSETS)
                );
            }
        );

        $container->addService(
            MaybeScheduleAttachmentDuplication::class,
            static function (Container $container): MaybeScheduleAttachmentDuplication {
                return new MaybeScheduleAttachmentDuplication(
                    $container[ServerRequest::class],
                    $container[AttachmentDuplicatorScheduler::class]
                );
            }
        );

        /* -----------------------------------------------------------------------------
           AttachmentDuplicationProcessAction
           -------------------------------------------------------------------------- */

        $container->addService(
            self::SCHEDULE_ACTION_ATTACHMENTS_REMOVER_SERVICE,
            static function (Container $container): RemoveActionTask {
                return new RemoveActionTask(
                    $container[ServerRequest::class],
                    $container[Scheduler::class],
                    ScheduleAssetManager::NAME_ATTACHMENT_SCHEDULE_ID,
                    AttachmentDuplicatorScheduler::SCHEDULE_HOOK
                );
            }
        );

        $container->addService(
            RemoveAttachmentIdsTask::class,
            static function (Container $container): RemoveAttachmentIdsTask {
                return new RemoveAttachmentIdsTask(
                    $container[ServerRequest::class],
                    $container[SiteScheduleOption::class],
                    ScheduleAssetManager::NAME_SITE_ID
                );
            }
        );

        $container->addService(
            self::SITE_DUPLICATION_ACTIONS,
            static function (Container $container): array {
                return apply_filters(
                    self::FILTER_SITE_DUPLICATION_ACTIONS,
                    [
                        ScheduleActionsNames::STOP_ATTACHMENTS_COPY => [
                            $container[self::SCHEDULE_ACTION_ATTACHMENTS_REMOVER_SERVICE],
                            $container[RemoveAttachmentIdsTask::class],
                        ],
                    ]
                );
            }
        );

        $container->share(
            self::SITE_DUPLICATION_SUCCESS_ACTIONS_MESSAGES,
            static function (): array {
                return apply_filters(
                    self::FILTER_SUCCESS_ACTIONS_MESSAGES,
                    [
                        ScheduleActionsNames::STOP_ATTACHMENTS_COPY => esc_html_x(
                            'Attachments copy has been stopped without errors.',
                            'Site Duplication',
                            'multilingualpress'
                        ),
                    ]
                );
            }
        );

        $container->addService(
            self::SCHEDULE_ACTION_ATTACHMENT_HANDLER_SERVICE,
            static function (Container $container): ScheduleActionRequestHandler {
                return new ScheduleActionRequestHandler(
                    $container[NonceFactory::class]->create(
                        [self::SCHEDULE_ACTION_ATTACHMENTS_NONCE_KEY]
                    ),
                    $container->get(self::SITE_DUPLICATION_ACTIONS),
                    $container->get(MessageFactoryInterface::class),
                    $container->get(Context::class),
                    $container->get(self::SITE_DUPLICATION_SUCCESS_ACTIONS_MESSAGES),
                    self::SCHEDULE_ACTION_ATTACHMENTS_AJAX_HOOK_NAME,
                    self::SCHEDULE_ACTION_ATTACHMENTS_USER_REQUIRED_CAPABILITY
                );
            }
        );
    }

    /**
     * @inheritdoc
     */
    public function bootstrap(Container $container)
    {
        $scheduleActionRequestHandler = $container->get(self::SCHEDULE_ACTION_ATTACHMENT_HANDLER_SERVICE);
        $attachmentDuplicatorHandler = $container->get(AttachmentDuplicatorHandler::class);
        $tableList = $container->get(TableList::class);

        $settingView = SiteSettingMultiView::fromViewModels(
            [
                $container->get(BasedOnSiteSetting::class),
                $container->get(CopyAttachmentsSetting::class),
                $container->get(ConnectContentSetting::class),
                $container->get(ConnectCommentsSetting::class),
                $container->get(ActivatePluginsSetting::class),
                $container->get(CopyUsersSetting::class),
                $container->get(SearchEngineVisibilitySetting::class),
            ]
        );

        $this->duplicateSiteBackCompactBootstrap($container);
        $this->defineInitialSettingsBackCompactBootstrap($container);
        $this->excludeMlpTables($tableList->mlpTableNames());

        add_action(
            SiteSettingsSectionView::ACTION_AFTER . '_' . NewSiteSettings::SECTION_ID,
            [$settingView, 'render']
        );

        add_action('admin_footer', [new NewSiteScheduleTemplate(), 'render']);

        add_action(
            SiteDuplicator::DUPLICATE_ACTION_KEY,
            [
                $container->get(MaybeScheduleAttachmentDuplication::class),
                'maybeScheduleAttachmentsDuplication',
            ],
            10,
            2
        );

        add_action(AttachmentDuplicatorScheduler::SCHEDULE_HOOK, [$attachmentDuplicatorHandler, 'handle']);

        add_action(
            'wp_ajax_' . self::SCHEDULE_ACTION_ATTACHMENTS_AJAX_HOOK_NAME,
            static function () use ($container, $scheduleActionRequestHandler) {
                $scheduleActionRequestHandler->handle($container[ServerRequest::class]);
            }
        );

        /**
         * Enqueues the scripts for scheduling the assets duplication.
         */
        add_action('network_site_new_form', [$container->get(ScheduleAssetManager::class), 'enqueueScript']);
    }

    /**
     * @param Container $container
     * @throws Throwable
     */
    protected function duplicateSiteBackCompactBootstrap(Container $container)
    {
        $wpVersion = wpVersion();
        $siteDuplicator = $container->get(SiteDuplicator::class);

        if (version_compare($wpVersion, '5.1', '<')) {
            add_action('wpmu_new_blog', wpHookProxy([$siteDuplicator, 'duplicateSite']), 0);
            return;
        }

        add_action(
            'wp_initialize_site',
            static function (WP_Site $wpSite) use ($siteDuplicator) {
                $siteId = (int)$wpSite->blog_id;
                ($siteId > 0) and $siteDuplicator->duplicateSite($siteId);
            },
            20
        );
    }

    /**
     * @param Container $container
     * @throws LateAccessToNotSharedService
     * @throws NameNotFound
     * @throws Throwable
     */
    protected function defineInitialSettingsBackCompactBootstrap(Container $container)
    {
        $wpVersion = wpVersion();
        $siteSettingsUpdater = $container->get(SiteSettingsUpdater::class);

        if (version_compare($wpVersion, '5.1', '<')) {
            add_action(
                'wpmu_new_blog',
                wpHookProxy([$siteSettingsUpdater, 'defineInitialSettings'])
            );

            return;
        }

        add_action(
            'wp_initialize_site',
            static function (WP_Site $wpSite) use ($siteSettingsUpdater) {
                $siteId = (int)$wpSite->blog_id;
                ($siteId > 0) and $siteSettingsUpdater->defineInitialSettings($siteId);
            },
            20
        );
    }

    /**
     * @param Container $container
     * @return Nonce
     */
    private function duplicateNonce(Container $container): Nonce
    {
        $nonce = $container[NonceFactory::class]->create(['duplicate_site']);
        // When creating a new site, its ID is not yet known, so create a nonce for a fixed site ID 0.
        if ($nonce instanceof SiteAwareNonce) {
            $nonce = $nonce->withSite(0);
        }

        return $nonce;
    }

    /**
     * Adds all MLP tables to site duplication excluded tables list.
     *
     * @param string[] $mlpTableNames The list of all MLP table names.
     */
    private function excludeMlpTables(array $mlpTableNames): void
    {
        add_filter(
            SiteDuplicator::FILTER_EXCLUDED_TABLES,
            static function () use ($mlpTableNames): array {
                return $mlpTableNames;
            }
        );
    }
}
