<?php

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\SiteDuplication\Schedule;

use Inpsyde\Assets\Asset;
use Inpsyde\Assets\AssetManager;
use Inpsyde\Assets\Script;
use Inpsyde\MultilingualPress\Framework\Factory\NonceFactory;
use Inpsyde\MultilingualPress\Framework\Nonce\Nonce;
use Inpsyde\MultilingualPress\Schedule\AjaxScheduleHandler;
use Inpsyde\MultilingualPress\SiteDuplication\ServiceProvider;

class ScheduleAssetManager
{
    public const NAME_ATTACHMENT_SCHEDULE_ID = 'scheduleId';
    public const NAME_SITE_ID = 'siteId';
    public const SCRIPTS_HANDLER_NAME = 'multilingualpress-site-duplication-admin';

    /**
     * @var SiteScheduleOption
     */
    protected $siteScheduleOption;

    /**
     * @var AjaxScheduleHandler
     */
    protected $ajaxScheduleHandler;

    /**
     * @var NonceFactory
     */
    protected $scheduleActionsNonce;

    /**
     * @var string
     */
    protected $urlToAssetsFolder;

    public function __construct(
        SiteScheduleOption $siteScheduleOption,
        AjaxScheduleHandler $ajaxScheduleHandler,
        Nonce $scheduleActionsNonce,
        string $urlToAssetsFolder
    ) {

        $this->siteScheduleOption = $siteScheduleOption;
        $this->ajaxScheduleHandler = $ajaxScheduleHandler;
        $this->scheduleActionsNonce = $scheduleActionsNonce;
        $this->urlToAssetsFolder = $urlToAssetsFolder;
    }

    /**
     * Enqueues the scripts for scheduling the assets duplication.
     *
     * @return void
     */
    public function enqueueScript()
    {
        $scheduleUrl = $this->scheduleUrl();
        $attachmentDuplicatorTranslations = $this->attachmentDuplicatorTranslations();
        $attachmentDuplicatorActions = $this->attachmentDuplicatorActions();
        $urlToAssetsFolder = $this->urlToAssetsFolder;

        add_action(
            AssetManager::ACTION_SETUP,
            static function (AssetManager $assetManager) use (
                $urlToAssetsFolder,
                $scheduleUrl,
                $attachmentDuplicatorTranslations,
                $attachmentDuplicatorActions
            ) {

                $script = new Script(self::SCRIPTS_HANDLER_NAME, "{$urlToAssetsFolder}js/site-duplication-admin.min.js");
                $script
                    ->withLocalize('siteDuplicatorScheduleData', [
                        'ajaxUrl' => esc_url(admin_url('admin-ajax.php')),
                        'scheduleUrl' => $scheduleUrl,
                        'attachmentDuplicatorTranslations' => $attachmentDuplicatorTranslations,
                        'attachmentDuplicatorActions' => $attachmentDuplicatorActions,
                    ])
                    ->forLocation(Asset::BACKEND);

                $assetManager->register($script);
            }
        );
    }

    /**
     * Retrieve the ajax schedule information url to call to obtain information about the current
     * status of the cron jobs
     *
     * @return string
     */
    protected function scheduleUrl(): string
    {
        $allSchedule = $this->siteScheduleOption->allSchedule();
        $schedule = reset($allSchedule) ?: '';
        return $this->ajaxScheduleHandler->scheduleInfoAjaxUrl($schedule);
    }

    /**
     * @return array
     */
    protected function attachmentDuplicatorTranslations(): array
    {
        return [
            'doNotCloseBrowserTabMessage' => esc_html__(
                'You cannot close this window until the entire process is finished.',
                'multilingualpress'
            ),
            'scheduleInfoErrorMessage' => esc_html__(
                'Something went wrong when trying to retrieve information about media copy task. Please, wait a minute or try to reload the page. If the problem persists please contact our support.',
                'multilingualpress'
            ),
            'preventFormSubmissionMessage' => esc_html__(
                'Actually one or more scheduled jobs are running, please wait until everything has been completed.',
                'multilingualpress'
            ),
        ];
    }

    /**
     * @return array
     */
    protected function attachmentDuplicatorActions(): array
    {
        return [
            'nonce' => (string)$this->scheduleActionsNonce,
            'nonceAction' => $this->scheduleActionsNonce->action(),
            'nameAjaxHook' => ServiceProvider::SCHEDULE_ACTION_ATTACHMENTS_AJAX_HOOK_NAME,
        ];
    }
}
