<?php

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\Onboarding;

use Inpsyde\MultilingualPress\Framework\Admin\AdminNotice;
use Inpsyde\MultilingualPress\Framework\Api\SiteRelations;
use Inpsyde\MultilingualPress\Framework\Database\Exception\NonexistentTable;
use Inpsyde\MultilingualPress\Framework\Http\Request;

class Onboarding
{
    public const OPTION_ONBOARDING_DISMISSED = 'onboarding_dismissed';

    /**
     * @var Notice
     */
    private $onboardingMessages;

    /**
     * @var State
     */
    private $onboardingState;

    /**
     * @var Request
     */
    private $request;

    /**
     * @var SiteRelations
     */
    private $siteRelations;

    public function __construct(
        SiteRelations $siteRelations,
        Request $request,
        State $onboardingState,
        Notice $onboardingMessages
    ) {

        $this->siteRelations = $siteRelations;
        $this->request = $request;
        $this->onboardingState = $onboardingState;
        $this->onboardingMessages = $onboardingMessages;
    }

    /**
     * @return void
     * @throws NonexistentTable
     */
    public function messages()
    {
        if (!$this->mayDisplayMessage()) {
            return;
        }

        $siteRelations = $this->siteRelations->allRelations();
        $onboardingState = $this->onboardingState->update(
            $this->onboardingState->read(),
            $siteRelations
        );
        $messageContent = $this->onboardingMessages->onboardingMessageContent(
            $onboardingState
        );

        AdminNotice::multilingualpress($messageContent->message)
            ->withTitle($messageContent->title)
            ->makeDismissible()
            ->inAllScreens()
            ->render();
    }

    /**
     * @return void
     */
    public function handleDismissOnboardingMessage()
    {
        $onboardingDismissed = $this->request->bodyValue(
            self::OPTION_ONBOARDING_DISMISSED,
            INPUT_GET,
            FILTER_SANITIZE_SPECIAL_CHARS
        );

        if ($onboardingDismissed === '1' && current_user_can('create_sites')) {
            update_site_option(self::OPTION_ONBOARDING_DISMISSED, true);
        }
    }

    /**
     * @return void
     */
    public function handleAjaxDismissOnboardingMessage()
    {
        if (!wp_doing_ajax()) {
            return;
        }

        if (!doing_action('wp_ajax_onboarding_plugin')) {
            wp_send_json_error('Invalid action.');
        }

        if (update_site_option(self::OPTION_ONBOARDING_DISMISSED, true)) {
            wp_send_json_success();
            die;
        }

        wp_send_json_error('Not updated.');
    }

    /**
     * @return bool
     */
    private function mayDisplayMessage(): bool
    {
        if (!current_user_can('create_sites')) {
            return false;
        }

        if ((bool)get_site_option(self::OPTION_ONBOARDING_DISMISSED) === true) {
            return false;
        }

        return true;
    }
}
