<?php

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\Core\Admin;

use Inpsyde\MultilingualPress\Framework\Admin\AdminNotice;
use Inpsyde\MultilingualPress\Framework\Admin\PersistentAdminNotices;
use Inpsyde\MultilingualPress\Framework\Http\Request;
use Inpsyde\MultilingualPress\Framework\Nonce\Nonce;
use Inpsyde\MultilingualPress\Framework\Setting\SiteSettingsUpdatable;

use function Inpsyde\MultilingualPress\redirectAfterSettingsUpdate;

/**
 * Class PostTypeSlugsSettingsUpdateRequestHandler
 */
class PostTypeSlugsSettingsUpdateRequestHandler
{
    const ACTION = 'update_multilingualpress_post_type_slugs_site_settings';

    /**
     * @var Nonce
     */
    private $nonce;

    /**
     * @var Request
     */
    private $request;

    /**
     * @var PersistentAdminNotices
     */
    private $notice;

    /**
     * @var SiteSettingsUpdatable
     */
    private $updater;

    /**
     * @param PostTypeSlugsSettingsUpdater $updater
     * @param Request $request
     * @param PersistentAdminNotices $notice
     * @param Nonce $nonce
     */
    public function __construct(
        PostTypeSlugsSettingsUpdater $updater,
        Request $request,
        PersistentAdminNotices $notice,
        Nonce $nonce
    ) {

        $this->updater = $updater;
        $this->request = $request;
        $this->notice = $notice;
        $this->nonce = $nonce;
    }

    /**
     * Handles POST requests.
     */
    public function handlePostRequest()
    {
        if (!$this->nonce->isValid()) {
            wp_die('Invalid', 'Invalid', 403);
        }

        $siteId = (int)$this->request->bodyValue('id', INPUT_POST, FILTER_SANITIZE_NUMBER_INT);
        if (!$siteId) {
            wp_die('Invalid site', 'Invalid site', 403);
        }

        $this->updater->updateSettings($siteId);
        $this->onSlugsUpdatedNotice($siteId);
        redirectAfterSettingsUpdate();
    }

    /**
     * @param int $siteId
     */
    private function onSlugsUpdatedNotice(int $siteId): void
    {

        $permalinkOptionsUrl = get_admin_url($siteId, 'options-permalink.php');

        // Translators: %s will be replaced with the HTML anchor tag for the permalink settings page.
        $messageFormat = __(
            'Please, update the %s to reflect the updated slugs.',
            'multilingualpress'
        );

        $anchorTag = sprintf(
            '<a href="%s">%s</a>',
            esc_url($permalinkOptionsUrl),
            __('permalink settings', 'multilingualpress')
        );

        $message = sprintf($messageFormat, $anchorTag);

        $this->notice->add(AdminNotice::warning($message)->inNetworkScreens());
    }
}
