<?php

declare(strict_types=1);

use Inpsyde\MultilingualPress\Core\ServiceProvider as CoreServiceProvider;
use Inpsyde\MultilingualPress\Framework\Admin\SettingsPageTab;
use Inpsyde\MultilingualPress\Framework\Admin\SettingsPageTabData;
use Inpsyde\MultilingualPress\Framework\Factory\NonceFactory;
use Inpsyde\MultilingualPress\Framework\Module\ModuleManager;
use Inpsyde\MultilingualPress\Framework\Service\Container;
use Inpsyde\MultilingualPress\Framework\Setting\SettingOption;
use Inpsyde\MultilingualPress\Module\Redirect\Redirector\Redirector;
use Inpsyde\MultilingualPress\Module\Redirect\ServiceProvider as RedirectModule;
use Inpsyde\MultilingualPress\Module\Redirect\Settings\RedirectSiteSettings;
use Inpsyde\MultilingualPress\Module\Redirect\Settings\RedirectUserSetting;
use Inpsyde\MultilingualPress\Module\Redirect\Settings\Renderers\RedirectFallbackViewRenderer;
use Inpsyde\MultilingualPress\Module\Redirect\Settings\Renderers\RedirectExecutionTypeViewRenderer;
use Inpsyde\MultilingualPress\Module\Redirect\Settings\Renderers\RedirectTypeViewRenderer;
use Inpsyde\MultilingualPress\Module\Redirect\Settings\Repository\Repository;
use Inpsyde\MultilingualPress\Module\Redirect\Settings\TabView;
use Inpsyde\MultilingualPress\Module\Redirect\Settings\Updater;

(static function (Container $container) {

    $container->share(
        Repository::class,
        static function (): Repository {
            return new Repository();
        }
    );

    $container->share(
        'redirect_site_settings',
        static function (Container $container): array {
            $repository = $container->get(Repository::class);
            return [
                new SettingOption(
                    $repository::OPTION_SITE_ENABLE_REDIRECT,
                    $repository::OPTION_SITE_ENABLE_REDIRECT,
                    __('Enable automatic redirect', 'multilingualpress'),
                    ''
                ),
                new SettingOption(
                    $repository::OPTION_SITE_ENABLE_REDIRECT_FALLBACK,
                    $repository::OPTION_SITE_ENABLE_REDIRECT_FALLBACK,
                    __('Redirect Fallback site for this language', 'multilingualpress'),
                    ''
                ),
            ];
        }
    );

    $container->share(
        RedirectSiteSettings::class,
        static function (Container $container): RedirectSiteSettings {
            return new RedirectSiteSettings(
                $container->get('redirect_site_settings'),
                $container->get(NonceFactory::class)->create([RedirectModule::SETTING_NONCE_ACTION . 'site']),
                $container->get(Repository::class)
            );
        }
    );

    $container->share(
        RedirectUserSetting::class,
        static function (Container $container): RedirectUserSetting {
            return new RedirectUserSetting(
                Repository::META_KEY_USER,
                $container->get(NonceFactory::class)->create([RedirectModule::SETTING_NONCE_ACTION . 'user']),
                $container->get(Repository::class)
            );
        }
    );

    $container->share(
        Updater::class,
        static function (Container $container): Updater {
            return new Updater(
                $container->get(NonceFactory::class)->create(['save_module_redirect_settings']),
                $container->get(Repository::class)
            );
        }
    );

    /**
     * Configuration for the redirect types.
     *
     * @return array<string, string> The map of redirect types to their titles.
     * phpcs:disable Inpsyde.CodeQuality.LineLength.TooLong
     */
    $container->share(
        RedirectModule::CONFIGURATION_NAME_FOR_REDIRECT_TYPES,
        static function (): array {
            return [
                Redirector::REDIRECT_TYPE_BROWSER_LANGUAGE => __('Browser language (Header)', 'multilingualpress'),
                Redirector::REDIRECT_TYPE_GEOLOCATION => __('Geolocation (IP Address)', 'multilingualpress'),
                Redirector::REDIRECT_TYPE_GEOLOCATION_WITH_FALLBACK_TO_BROWSER_LANGUAGE => __('Geolocation with fallback to browser Language', 'multilingualpress'),
            ];
            // phpcs:enable
        }
    );

    $settingsRepository = $container->get(Repository::class);
    $nonceFactory = $container->get(NonceFactory::class);
    $moduleManager = $container->get(ModuleManager::class);
    $redirectTypes = $container->get(RedirectModule::CONFIGURATION_NAME_FOR_REDIRECT_TYPES);

    /**
     * Filters the MLP network settings and ads the tab for this module.
     */
    add_filter(
        CoreServiceProvider::ACTION_BUILD_TABS,
        static function (array $tabs) use ($settingsRepository, $nonceFactory, $moduleManager, $redirectTypes): array {
            if (!$moduleManager->isModuleActive(RedirectModule::MODULE_ID)) {
                return $tabs;
            }

            $tabs['redirect'] = new SettingsPageTab(
                new SettingsPageTabData('redirect', __('Redirect', 'multilingualpress'), 'redirect'),
                new TabView(
                    $nonceFactory->create(['save_module_redirect_settings']),
                    new RedirectFallbackViewRenderer($settingsRepository),
                    new RedirectExecutionTypeViewRenderer(
                        esc_html_x('Redirect Execution', 'Redirect Module Settings', 'multilingualpress'),
                        Repository::MODULE_SETTING_EXECUTION_TYPE,
                        esc_html_x(
                            'Choose which programing language should be used for the redirection.',
                            'Redirect Module Settings',
                            'multilingualpress'
                        ),
                        $settingsRepository,
                        [
                            Redirector::TYPE_PHP => esc_html_x(
                                'Server-side redirect (PHP)',
                                'Redirect Module Settings',
                                'multilingualpress'
                            ),
                            Redirector::TYPE_JAVASCRIPT => esc_html_x(
                                'Client-side redirect (JavaScript)',
                                'Redirect Module Settings',
                                'multilingualpress'
                            ),
                        ]
                    ),
                    new RedirectTypeViewRenderer(
                        esc_html_x('Redirect Type', 'Redirect Module Settings', 'multilingualpress'),
                        Repository::MODULE_SETTING_REDIRECT_TYPE,
                        esc_html_x(
                            'Choose how the redirection should work.',
                            'Redirect Module Settings',
                            'multilingualpress'
                        ),
                        $redirectTypes,
                        $settingsRepository
                    )
                )
            );

            return $tabs;
        }
    );
}
)($container); //phpcs:disable VariableAnalysis.CodeAnalysis.VariableAnalysis.UndefinedVariable
