<?php

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\Framework\Http;

use Inpsyde\MultilingualPress\Framework\Url\Url;

/**
 * Interface for all HTTP request abstraction implementations.
 */
interface Request
{
    public const CONNECT = 'CONNECT';
    public const DELETE = 'DELETE';
    public const GET = 'GET';
    public const HEAD = 'HEAD';
    public const OPTIONS = 'OPTIONS';
    public const PATCH = 'PATCH';
    public const POST = 'POST';
    public const PUT = 'PUT';
    public const TRACE = 'TRACE';

    public const INPUT_GET = INPUT_GET;
    public const INPUT_POST = INPUT_POST;
    public const INPUT_REQUEST = 99;
    public const INPUT_COOKIE = INPUT_COOKIE;
    public const INPUT_SERVER = INPUT_SERVER;
    public const INPUT_ENV = INPUT_ENV;

    public const METHODS = [
        self::CONNECT,
        self::DELETE,
        self::GET,
        self::HEAD,
        self::OPTIONS,
        self::PATCH,
        self::POST,
        self::PUT,
        self::TRACE,
    ];

    /**
     * Returns the URL for current request.
     *
     * @return Url
     */
    public function url(): Url;

    /**
     * Returns the body of the request as string.
     *
     * @return string
     */
    public function body(): string;

    /**
     * Return a value from request body, optionally filtered.
     *
     * @param string $name
     * @param int $source The input source of the value. One of the `INPUT_*` constants.
     * @param int $filter
     * @param int $options
     * @return mixed
     */
    public function bodyValue(
        string $name,
        int $source = self::INPUT_REQUEST,
        int $filter = FILTER_UNSAFE_RAW,
        int $options = FILTER_FLAG_NONE
    );

    /**
     * Returns header value as set in the request.
     *
     * @param string $name
     * @return string
     */
    public function header(string $name): string;

    /**
     * Returns method (GET, POST..) value as set in the request.
     *
     * @return string
     */
    public function method(): string;
}
