<?php // phpcs:ignore
/*
 * @wordpress-plugin
 * Plugin Name: MultilingualPress
 * Plugin URI: https://multilingualpress.org/
 * Description: The multisite-based plugin for your multilingual WordPress websites.
 * Author: Inpsyde GmbH
 * Author URI: https://inpsyde.com
 * Version: 4.7.0
 * Text Domain: multilingualpress
 * Domain Path: /languages/
 * License: GPLv2+
 * Network: true
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * 
 */

namespace Inpsyde\MultilingualPress;

// phpcs:disable PSR1.Files.SideEffects
// phpcs:disable NeutronStandard.StrictTypes.RequireStrictTypes
// phpcs:disable Inpsyde.CodeQuality.ReturnTypeDeclaration
// phpcs:disable Inpsyde.CodeQuality.FunctionLength.TooLong

use Exception;

defined('ABSPATH') or die();

/**
 * @param $function
 *
 * phpcs:disable Inpsyde.CodeQuality.ArgumentTypeDeclaration.NoArgumentType
 */
function deactivateNotice($function, bool $deactivate = true)
{
    if (!is_callable($function)) {
        return;
    }
    $hooks = [
        'admin_notices',
        'network_admin_notices',
    ];
    foreach ($hooks as $hook) {
        add_action($hook, static function () use ($function, $deactivate) {
            $function();

            if ($deactivate) {
                deactivate_plugins(plugin_basename(__FILE__));

                // Suppress the "Plugin activated" notice.
                unset($_GET['activate']); // phpcs:ignore
            }
        });
    }
}

/** @noinspection ConstantCanBeUsedInspection */
if (version_compare(PHP_VERSION, '7.4', '<')) {
    deactivateNotice(static function () {
        $message = __(
            'MultilingualPress requires at least  PHP version 7.4. <br />Please ask your server administrator to update your environment to PHP version 7.4.',
            'multilingualpress'
        );

        printf(
            '<div class="notice notice-error"><span class="notice-title">%1$s</span><p>%2$s</p></div>',
            esc_html__(
                'The plugin MultilingualPress has been deactivated',
                'multilingualpress'
            ),
            wp_kses($message, ['br' => true])
        );
    });
    return;
}

if (!function_exists('is_plugin_active')) {
    /** @noinspection PhpIncludeInspection */
    require_once untrailingslashit(ABSPATH) . '/wp-admin/includes/plugin.php';
}
if (is_plugin_active('multilingual-press/multilingual-press.php')) {
    deactivateNotice(static function () {
        $message = __(
            'You try to activate MLP Version 3+ but you already have MLP Version 2 activated. Please check out our <a href="https://multilingualpress.org/docs/getting-started-with-multilingualpress-3/">guide</a> before you continue!',
            'multilingualpress'
        );

        printf(
            '<div class="notice notice-error"><span class="notice-title">%1$s</span><p>%2$s</p></div>',
            esc_html__(
                'The plugin MultilingualPress has been deactivated',
                'multilingualpress'
            ),
            wp_kses($message, ['a' => ['href' => true]])
        );
    });
    return;
}

/**
 * Adds the message in plugin list view for the next version.
 *
 * @todo Remove this after the release.
 */
$pluginBasename = plugin_basename(__FILE__);
add_action("in_plugin_update_message-{$pluginBasename}", static function (): void {

    $title = __('Security update - Future MultilingualPress versions will require PHP 7.4', 'multilingualpress');
    $message = __('The next MultilingualPress update will change
        the PHP version requirements from PHP 7.2 to PHP 7.4.</ br>
        Following MultilingualPress 4.7.0, PHP versions older than PHP 7.4 will no longer be supported.</ br>
        Read more about the change in the <a target="_blank" href="https://multilingualpress.org/docs/php-7-4-update/">documentation here</a>.', 'multilingualpress');

    ?>
    </br>
    </br>
    </br>
    <strong><?= esc_html($title);?></strong>
    </br>
    </br>
    <?= wp_kses_post($message);?>
    <?php
}, 100);

/**
 * Updates the URL for the "View details" link on plugins page.
 *
 * @param array $links An array of plugin action links.
 * @param string $file Path to the plugin file relative to the plugins directory.
 * @return array Modified array of plugin action links.
 */
function updatePluginDetailsUrl(array $links, string $file): array
{

    $pluginBasename = plugin_basename(__FILE__);

    if ($file === $pluginBasename) {
        unset($links[2]);
        $links[] = sprintf(
            '<a href="%s" target="_blank">%s</a>',
            'https://multilingualpress.org/',
            __('Visit plugin site', 'multilingualpress')
        );
    }

    return $links;
}

add_filter('plugin_row_meta', __NAMESPACE__ . '\\updatePluginDetailsUrl', 10, 2);

/**
 * Declares HPOS compat.
 * @see https://github.com/woocommerce/woocommerce/wiki/High-Performance-Order-Storage-Upgrade-Recipe-Book#declaring-extension-incompatibility
 */
add_action('before_woocommerce_init', static function () {
    if (!class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        return;
    }

    \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
});

/**
 * Loads definitions and/or autoloader.
 *
 * @param string $rootDir
 * @throws Exception
 */
function autoload(string $rootDir)
{
    $autoloadPath = "$rootDir/vendor/autoload.php";
    if (file_exists($autoloadPath)) {
        /* @noinspection PhpIncludeInspection */
        require_once $autoloadPath;
    }

    /* @noinspection PhpIncludeInspection */
    require_once "$rootDir/src/inc/functions.php";
}

/**
 * Bootstraps MultilingualPress.
 *
 * @return bool
 *
 * @wp-hook plugins_loaded
 * @throws \Throwable
 * @return bool
 */
function bootstrap()
{
    $bootstrapped = false;

    try {
        /** @var Framework\Service\Container $container */
        $container = resolve(null);
        $container = $container->shareValue(
            Framework\PluginProperties::class,
            new Framework\PluginProperties(__FILE__)
        );

        $providers = new Framework\Service\ServiceProvidersCollection();
        $modules = [
            new Activation\ServiceProvider(),
            new WpCli\ServiceProvider(),
            new Api\ServiceProvider(),
            new Cache\ServiceProvider(),
            new Core\ServiceProvider(),
            new Auth\ServiceProvider(),
            new Module\Blocks\ServiceProvider(),
            new Database\ServiceProvider(),
            new Factory\ServiceProvider(),
            new Installation\ServiceProvider(),
            new NavMenu\ServiceProvider(),
            new SiteDuplication\ServiceProvider(),
            new TranslationUi\ServiceProvider(),
            new Translator\ServiceProvider(),
            new Module\AltLanguageTitleInAdminBar\ServiceProvider(),
            new Module\Redirect\ServiceProvider(),
            new Module\Trasher\ServiceProvider(),
            new Module\LanguageManager\ServiceProvider(),
            new Module\LanguageSwitcher\ServiceProvider(),
            new Module\WooCommerce\ServiceProvider(),
            new Module\QuickLinks\ServiceProvider(),
            new Onboarding\ServiceProvider(),
            new Schedule\ServiceProvider(),
            new Customizer\ServiceProvider(),
            new Module\ACF\ServiceProvider(),
            new License\ServiceProvider(),
            new Module\BeaverBuilder\ServiceProvider(),
            new Module\Elementor\ServiceProvider(),
            new Module\User\ServiceProvider(),
            new Module\OriginalTranslationLanguage\ServiceProvider(),
            new Module\Comments\ServiceProvider(),
            new Module\ExternalSites\ServiceProvider(),
            new MediaLibrary\ServiceProvider(),
        ];

        $modules = apply_filters('multilingualpress.modules', $modules);

        foreach ($modules as $module) {
            $providers->add($module);
        }

        $multilingualpress = new MultilingualPress($container, $providers);

        /**
         * Fires right before MultilingualPress gets bootstrapped.
         *
         * Hook here to add custom service providers via
         * `ServiceProviderCollection::add_service_provider()`.
         *
         * @param Framework\Service\ServiceProvidersCollection $providers
         */
        do_action(ACTION_ADD_SERVICE_PROVIDERS, $providers);

        $bootstrapped = $multilingualpress->bootstrap();

        unset($providers);
    } catch (\Throwable $thr) {
        do_action(ACTION_LOG, 'error', $thr->getMessage(), compact('thr'));

        if (defined('WP_DEBUG') && WP_DEBUG) {
            throw $thr;
        }
    }

    return $bootstrapped;
}

/**
 * Triggers a plugin-specific activation action third parties can listen to.
 *
 * @wp-hook activate_{$plugin}
 */
function activate()
{
    /**
     * Fires when MultilingualPress is about to be activated.
     */
    do_action(ACTION_ACTIVATION);

    add_action(
        'activated_plugin',
        static function (string $plugin) {
            if (plugin_basename(__FILE__) === $plugin) {
                // Bootstrap MultilingualPress to take care of installation or upgrade routines.
                bootstrap();
            }
        },
        0
    );
}

/**
 * Load missed WordPress functions.
 */
function loadWordPressFunctions()
{
    if (!function_exists('wp_get_available_translations')) {
        require_once(ABSPATH . 'wp-admin/includes/translation-install.php');
    }
}

(static function (string $rootFile) {
    $rootDir = dirname($rootFile);

    try {
        autoload($rootDir);

        $modularity = "{$rootDir}/src/inc/modularity.php";
        if (file_exists($modularity)) {
            $moduleActivator = require_once $modularity;
            $moduleActivator($rootDir);
        }

        add_action('plugins_loaded', __NAMESPACE__ . '\\bootstrap', 0);

        register_activation_hook($rootFile, __NAMESPACE__ . '\\activate');

        loadWordPressFunctions();
    } catch (Exception $exception) {
        deactivateNotice(static function () use ($exception) {
            printf(
                '<div class="notice notice-error"><span class="notice-title">%1$s</span><p>%2$s</p></div>',
                esc_html__(
                    'The plugin MultilingualPress has been deactivated',
                    'multilingualpress'
                ),
                wp_kses($exception->getMessage(), [])
            );
        });
        return;
    }
})(__FILE__);
