<?php

declare(strict_types=1);

use Inpsyde\MultilingualPress\Attachment\Copier;
use Inpsyde\MultilingualPress\Framework\Http\ServerRequest;
use Inpsyde\MultilingualPress\Framework\Service\Container;
use Inpsyde\MultilingualPress\MediaLibrary\BulkCopyAttachments\BulkCopyAttachmentsRequestHandler;
use Inpsyde\MultilingualPress\MediaLibrary\BulkCopyAttachments\BulkCopyAttachmentsSettingsView;
use Inpsyde\MultilingualPress\MediaLibrary\ServiceProvider as MediaLibraryModule;

use function Inpsyde\MultilingualPress\siteExists;

(static function (Container $container) {
    $bulkCopyAttachmentsSettingsView = $container->get(BulkCopyAttachmentsSettingsView::class);
    $bulkCopyAttachmentsRequestHandler = $container->get(BulkCopyAttachmentsRequestHandler::class);
    $multilingualPressBulkActions = $container->get(MediaLibraryModule::PARAMETER_CONFIG_BULK_ACTIONS);
    $attachmentsCopier = $container->get(Copier::class);
    $bulkActionsKeyPrefix = $container->get(MediaLibraryModule::PARAMETER_CONFIG_BULK_ACTIONS_KEY_PREFIX);
    $serverRequest = $container->get(ServerRequest::class);

    /**
     * Renders the MLP copy attachments settings when Media Library view is in "Grid Mode".
     *
     * This is done by JS AJAX, cause WordPress doesn't provide any way to inject the HTML.
     */
    add_action(
        'wp_ajax_' . BulkCopyAttachmentsSettingsView::ACTION,
        static function () use ($bulkCopyAttachmentsSettingsView): void {
            if (!wp_doing_ajax()) {
                return;
            }

            if (!doing_action('wp_ajax_' . BulkCopyAttachmentsSettingsView::ACTION)) {
                wp_send_json_error('Invalid action.');
            }

            ob_start();
            $bulkCopyAttachmentsSettingsView->render();
            $response = ob_get_clean();

            wp_send_json_success($response);
        }
    );

    /**
     * Handles the bulk copy attachments.
     */
    add_action(
        'wp_ajax_' . BulkCopyAttachmentsRequestHandler::ACTION,
        static function () use ($bulkCopyAttachmentsRequestHandler, $serverRequest): void {
            $bulkCopyAttachmentsRequestHandler->handle($serverRequest);
        }
    );

    /**
     * Adds the MLP bulk actions when the Media Library view is in "List Mode".
     */
    add_filter(
        'bulk_actions-upload',
        static function (array $bulkActions) use ($multilingualPressBulkActions): array {
            return array_merge($bulkActions, $multilingualPressBulkActions);
        }
    );

    /**
     * Handles the MLP bulk actions when the Media Library view is in "List Mode".
     */
    add_filter(
        'handle_bulk_actions-upload',
        static function (
            string $redirectUrl,
            string $action,
            array $attachmentIds
        ) use (
            $multilingualPressBulkActions,
            $attachmentsCopier,
            $bulkActionsKeyPrefix
        ): string {

            if (empty($attachmentIds) || !array_key_exists($action, $multilingualPressBulkActions)) {
                return $redirectUrl;
            }

            $siteId = (int)str_replace($bulkActionsKeyPrefix, '', $action);
            if (!siteExists($siteId)) {
                return $redirectUrl;
            }

            $attachmentsCopier->copyById(get_current_blog_id(), $siteId, $attachmentIds);

            return $redirectUrl;
        },
        10,
        3
    );
}
)($container); //phpcs:disable VariableAnalysis.CodeAnalysis.VariableAnalysis.UndefinedVariable
