<?php

# -*- coding: utf-8 -*-
/*
 * This file is part of the MultilingualPress package.
 *
 * (c) Inpsyde GmbH
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\Module\ACF;

use Inpsyde\MultilingualPress\Attachment\Copier;
use Inpsyde\MultilingualPress\Framework\Database\Exception\NonexistentTable;
use Inpsyde\MultilingualPress\Framework\Http\Request;
use Inpsyde\MultilingualPress\Module\ACF\TranslationUi\Post\MetaboxFields;
use Inpsyde\MultilingualPress\TranslationUi\Post\PostRelationSaveHelper;
use Inpsyde\MultilingualPress\TranslationUi\Post\RelationshipContext;

use function Inpsyde\MultilingualPress\translationIds;

class FieldCopier
{
    /**
     * ACF field Types
     */
    const FIELD_TYPE_GROUP = 'group';
    const FIELD_TYPE_REPEATER = 'repeater';
    const FIELD_TYPE_FLEXIBLE = 'flexible_content';
    const FIELD_TYPE_IMAGE = 'image';
    const FIELD_TYPE_GALLERY = 'gallery';
    const FIELD_TYPE_TAXONOMY = 'taxonomy';

    const FILE_FIELD_TYPES_FILTER = 'multilingualpress_acf_file_field_types_filter';
    const DEFAULT_FILE_FIELD_TYPES = ['file', 'video', 'image', 'application'];

    /**
     * ACF flexible field's layout key
     * It is used to exclude the key from sync keys
     */
    const FLEXIBLE_FIELD_LAYOUT_KEY = 'acf_fc_layout';

    /**
     * @var Copier
     */
    protected $copier;

    /**
     * @var array
     */
    private $acfFileFieldTypes;

    public function __construct(Copier $copier)
    {
        $this->copier = $copier;
        $this->acfFileFieldTypes = apply_filters(
            self::FILE_FIELD_TYPES_FILTER,
            self::DEFAULT_FILE_FIELD_TYPES
        );
    }

    /**
     * The Method is a callback for PostRelationSaveHelper::FILTER_SYNC_KEYS filter
     * It will receive the keys of the meta fields which should be synced and
     * will add the ACF field keys
     *
     * @param array $keysToSync the array of meta keys
     * where should be added the ACF field keys to be synced
     * @param RelationshipContext $context
     * @param Request $request
     * @return array the array of meta keys to be synced
     * @throws NonexistentTable
     */
    public function handleCopyACFFields(
        array $keysToSync,
        RelationshipContext $context,
        Request $request
    ): array {

        $multilingualpress = $request->bodyValue(
            'multilingualpress',
            INPUT_POST,
            FILTER_DEFAULT,
            FILTER_FORCE_ARRAY
        );

        $remoteSiteId = $context->remoteSiteId();
        $translation = $multilingualpress["site-{$remoteSiteId}"] ?? '';
        $copyAcfFieldsIsChecked = $translation[MetaboxFields::FIELD_COPY_ACF_FIELDS] ?? 0;
        $fields = get_field_objects();

        if (empty($translation) || !$copyAcfFieldsIsChecked || empty($fields)) {
            return $keysToSync;
        }

        $keysToSync = $this->addACFFieldKeys($fields, $keysToSync, $context);

        return $keysToSync;
    }

    /**
     * This method will receive the ACF fields and
     * will find the appropriate meta keys depending on field type
     *
     * @param array $fields the array of advanced custom fields
     * @param array $keys the array of meta keys
     * where should be added the ACF field keys to be synced
     * @param RelationshipContext $context
     * @return array the array of meta keys to be synced
     *
     * phpcs:disable Inpsyde.CodeQuality.NestingLevel.High
     * phpcs:disable Generic.Metrics.CyclomaticComplexity.TooHigh
     * @throws NonexistentTable
     */
    protected function addACFFieldKeys(array $fields, array $keys, RelationshipContext $context): array
    {
        // phpcs:enable
        foreach ($fields as $fieldKey => $field) {
            switch ($field['type']) {
                case self::FIELD_TYPE_GROUP:
                case self::FIELD_TYPE_REPEATER:
                case self::FIELD_TYPE_FLEXIBLE:
                    if (empty($field['value']) || empty($field['name'])) {
                        break;
                    }
                    $foundKeys = $this->recursivelyFindLayoutFieldKeys($field['value'], $field['name'], $context);
                    foreach ($foundKeys as $key => $value) {
                        $keys[] = $value;
                    }
                    $keys[] = $fieldKey;
                    break;
                case self::FIELD_TYPE_IMAGE:
                case in_array($field['type'], $this->acfFileFieldTypes, true):
                case self::FIELD_TYPE_GALLERY:
                    $keys[] = $fieldKey;
                    $this->handleFileTypeFieldsCopy(
                        (string)$field['type'],
                        (array)$field['value'],
                        $context,
                        $fieldKey
                    );
                    break;
                case self::FIELD_TYPE_TAXONOMY:
                    if (empty($field['value'])) {
                        break;
                    }
                    $keys[] = $fieldKey;
                    $this->handleTaxTypeFieldsCopy(
                        (string)$field['type'],
                        (array)$field['value'],
                        $context,
                        $fieldKey
                    );
                    break;
                default:
                    $keys[] = $fieldKey;
            }
        }

        return $keys;
    }

    /**
     * This Method will recursively loop over the layout fields and will generate the necessary keys
     *
     * @param array $array the array of fields
     * @param string $parentKey The key of the parent field to bind with the current key
     * @param RelationshipContext $context
     * @return array the array of the generated keys
     * @throws NonexistentTable
     */
    protected function recursivelyFindLayoutFieldKeys(
        array $array,
        string $parentKey,
        RelationshipContext $context
    ): array {

        $keys = [];
        foreach ($array as $key => $value) {
            $newKey = $parentKey . '_' . $key;
            $fieldType = isset($array[$key]['type']) && isset($array[$key]['filename'])
                ? $array[$key]['type']
                : '';

            $this->handleFileTypeFieldsCopy($fieldType, (array)$value, $context, $newKey);
            $this->handleTaxTypeFieldsCopy($fieldType, (array)$value, $context, $newKey);

            if (
                is_array($array[$key]) &&
                $fieldType !== self::FIELD_TYPE_IMAGE &&
                $fieldType !== self::FIELD_TYPE_GALLERY &&
                !in_array($fieldType, $this->acfFileFieldTypes, true)
            ) {
                $keys = array_merge($keys, $this->recursivelyFindLayoutFieldKeys($array[$key], $newKey, $context));
            }

            if ($key !== self::FLEXIBLE_FIELD_LAYOUT_KEY) {
                $keys[] = $newKey;
            }
        }

        return $keys;
    }

    /**
     * The method will handle the Taxonomy type fields copy process
     *
     * @param string $fieldType The ACF field type, should be image, gallery or file
     * @param array $fieldValue The ACF field value
     * @param RelationshipContext $context
     * @param string $fieldKey the ACF field key
     * @throws NonexistentTable
     */
    protected function handleTaxTypeFieldsCopy(
        string $fieldType,
        array $fieldValue,
        RelationshipContext $context,
        string $fieldKey
    ) {

        if ($fieldType !== self::FIELD_TYPE_TAXONOMY) {
            return;
        }

        $remoteSiteId = $context->remoteSiteId();
        $connectedTaxIds = [];
        foreach ($fieldValue as $taxId) {
            $translations = translationIds($taxId, 'term');
            if (empty($translations[$remoteSiteId])) {
                continue;
            }
            $connectedTaxIds[0][] = $translations[$remoteSiteId];
        }
        $this->filterRemoteFieldValues($connectedTaxIds, $fieldKey);
    }

    /**
     * The method will handle the file type fields(image, gallery, file) copy process
     *
     * @param string $fieldType The ACF field type, should be image, gallery or file
     * @param array $fieldValue The ACF field value
     * @param RelationshipContext $context
     * @param string $fieldKey the ACF field key
     */
    protected function handleFileTypeFieldsCopy(
        string $fieldType,
        array $fieldValue,
        RelationshipContext $context,
        string $fieldKey
    ) {

        if (empty($fieldType)) {
            $fieldType = $this->ensureAttachmentType($fieldValue);
        }

        if (
            $fieldType !== self::FIELD_TYPE_IMAGE &&
            !in_array($fieldType, $this->acfFileFieldTypes, true) &&
            $fieldType !== self::FIELD_TYPE_GALLERY
        ) {
            return;
        }

        $attachmentIds[] = $fieldValue['id'] ?? $fieldValue[0] ?? '';
        $remoteAttachmentIds = $this->copyAttachments($context, $attachmentIds);

        if ($fieldType === self::FIELD_TYPE_GALLERY && !empty($fieldValue)) {
            $attachmentIds = [];
            foreach ($fieldValue as $attachment) {
                $attachmentIds[] = $attachment['id'] ?? $attachment ?? '';
            }

            $remoteAttachmentIds[] = $this->copyAttachments($context, $attachmentIds);
        }
        $this->filterRemoteFieldValues($remoteAttachmentIds, $fieldKey);
    }

    /**
     * The Method will copy the attachments from source site to the remote site
     *
     * @param RelationshipContext $context
     * @param array $attachmentIds The attachment IDs which should be copied to remote entity
     * @return array array of the attachment IDs in remote site which are copied from source site
     */
    protected function copyAttachments(RelationshipContext $context, array $attachmentIds): array
    {
        if (empty($attachmentIds)) {
            return [];
        }
        $sourceSiteId = $context->sourceSiteId();
        $remoteSiteId = $context->remoteSiteId();

        return $remoteAttachmentIds = $this->copier->copyById(
            $sourceSiteId,
            $remoteSiteId,
            $attachmentIds
        );
    }

    /**
     * The Method will filter the values of the ACF fields in remote site and will replace them with
     * the correct attachment ids which are copied from source site
     *
     * @param array $values The values which should be replaced in remote site fields
     * @param string $filedKey The ACF field Key of the remote site
     * for which the value should be filtered
     */
    protected function filterRemoteFieldValues(array $values, string $filedKey)
    {
        add_filter(
            PostRelationSaveHelper::FILTER_METADATA,
            static function ($valuesToSync) use ($values, $filedKey) {
                if (!empty($valuesToSync) && !empty($valuesToSync[$filedKey])) {
                    $valuesToSync[$filedKey] = $values;
                }
                return $valuesToSync;
            },
            10,
            2
        );
    }

    /**
     * If the image or gallery type fields in ACF are forced to
     * return only the attachment id then the field type will not
     * be possible to determine and the field will be shown just as
     * key (ACF field key) => value (attachment id) pair. This method
     * Will help to catch these type of fields, reassign and return their field type.
     * in case there will be other then image/gallery field types with integer values the Copier class
     * will check that before copying.
     *
     * @param array $fieldValue The ACF field value
     * @return string file type of field, can be image or gallery
     */
    protected function ensureAttachmentType(array $fieldValue): string
    {
        if (empty(array_filter($fieldValue, 'is_int') === $fieldValue)) {
            return '';
        }

        if (sizeof($fieldValue) > 1) {
            return self::FIELD_TYPE_GALLERY;
        }

        return self::FIELD_TYPE_IMAGE;
    }
}
