<?php // phpcs:ignore
/*
 * This file is part of the MultilingualPress package.
 *
 * (c) Inpsyde GmbH
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 *
 * @wordpress-plugin
 * Plugin Name: MultilingualPress
 * Plugin URI: https://multilingualpress.org/
 * Description: The multisite-based plugin for your multilingual WordPress websites.
 * Author: Inpsyde GmbH
 * Author URI: https://inpsyde.com
 * Version: 3.7.2
 * Text Domain: multilingualpress
 * Domain Path: /languages/
 * License: GPLv2+
 * Network: true
 * Requires at least: 4.8
 * Requires PHP: 7.0
 * 
 */

namespace Inpsyde\MultilingualPress;

// phpcs:disable PSR1.Files.SideEffects
// phpcs:disable NeutronStandard.StrictTypes.RequireStrictTypes
// phpcs:disable Inpsyde.CodeQuality.ReturnTypeDeclaration

defined('ABSPATH') or die();

/**
 * @param $function
 *
 * phpcs:disable Inpsyde.CodeQuality.ArgumentTypeDeclaration.NoArgumentType
 */
function deactivateNotice($function)
{
    if (!is_callable($function)) {
        return;
    }
    $hooks = [
        'admin_notices',
        'network_admin_notices',
    ];
    foreach ($hooks as $hook) {
        add_action($hook, static function () use ($function) {
            $function();

            deactivate_plugins(plugin_basename(__FILE__));

            // Suppress the "Plugin activated" notice.
            unset($_GET['activate']); // phpcs:ignore
        });
    }
}

/** @noinspection ConstantCanBeUsedInspection */
if (version_compare(PHP_VERSION, '7', '<')) {
    deactivateNotice(static function () {
        $message = __(
            'MultilingualPress 3 requires at least  PHP version 7. <br />Please ask your server administrator to update your environment to PHP version 7.',
            'multilingualpress'
        );

        printf(
            '<div class="notice notice-error"><span class="notice-title">%1$s</span><p>%2$s</p></div>',
            esc_html__(
                'The plugin MultilingualPress has been deactivated',
                'multilingualpress'
            ),
            wp_kses($message, ['br' => true])
        );
    });
    return;
}

if (!function_exists('is_plugin_active')) {
    /** @noinspection PhpIncludeInspection */
    require_once untrailingslashit(ABSPATH) . '/wp-admin/includes/plugin.php';
}
if (is_plugin_active('multilingual-press/multilingual-press.php')) {
    deactivateNotice(static function () {
        $message = __(
            'You try to activate MLP Version 3 but you already have MLP Version 2 activated. Please check out our <a href="https://multilingualpress.org/docs/getting-started-with-multilingualpress-3/">guide</a> before you continue!',
            'multilingualpress'
        );

        printf(
            '<div class="notice notice-error"><span class="notice-title">%1$s</span><p>%2$s</p></div>',
            esc_html__(
                'The plugin MultilingualPress has been deactivated',
                'multilingualpress'
            ),
            wp_kses($message, ['a' => ['href' => true]])
        );
    });
    return;
}

/**
 * Loads definitions and/or autoloader.
 */
function autoload(string $rootDir)
{
    /* @noinspection PhpIncludeInspection */
    require_once "$rootDir/src/inc/functions.php";

    if (file_exists("$rootDir/vendor/autoload.php")) {
        /* @noinspection PhpIncludeInspection */
        require_once "$rootDir/vendor/autoload.php";
    }
}

/**
 * Bootstraps MultilingualPress.
 *
 * @return bool
 *
 * @wp-hook plugins_loaded
 * @throws \Throwable
 * @return bool
 */
function bootstrap()
{
    $bootstrapped = false;

    try {
        /** @var Framework\Service\Container $container */
        $container = resolve(null);
        $container = $container->shareValue(
            Framework\PluginProperties::class,
            new Framework\PluginProperties(__FILE__)
        );

        $providers = new Framework\Service\ServiceProvidersCollection();
        $providers
            ->add(new Activation\ServiceProvider())
            ->add(new Api\ServiceProvider())
            ->add(new Asset\ServiceProvider())
            ->add(new Cache\ServiceProvider())
            ->add(new Core\ServiceProvider())
            ->add(new Auth\ServiceProvider())
            ->add(new Database\ServiceProvider())
            ->add(new Factory\ServiceProvider())
            ->add(new Installation\ServiceProvider())
            ->add(new Integration\ServiceProvider())
            ->add(new NavMenu\ServiceProvider())
            ->add(new SiteDuplication\ServiceProvider())
            ->add(new TranslationUi\ServiceProvider())
            ->add(new Translator\ServiceProvider())
            ->add(new Module\AltLanguageTitleInAdminBar\ServiceProvider())
            ->add(new Module\Redirect\ServiceProvider())
            ->add(new Module\Trasher\ServiceProvider())
            ->add(new Module\LanguageManager\ServiceProvider())
            ->add(new Module\LanguageSwitcher\ServiceProvider())
            ->add(new Module\WooCommerce\ServiceProvider())
            ->add(new Module\QuickLinks\ServiceProvider())
            ->add(new Onboarding\ServiceProvider())
            ->add(new Schedule\ServiceProvider())
            ->add(new Customizer\ServiceProvider())
            ->add(new Module\ACF\ServiceProvider())
            ->add(new License\ServiceProvider())
            ->add(new Module\BeaverBuilder\ServiceProvider())
            ->add(new Module\Elementor\ServiceProvider())
            ->add(new Module\User\ServiceProvider());

        $multilingualpress = new MultilingualPress($container, $providers);

        /**
         * Fires right before MultilingualPress gets bootstrapped.
         *
         * Hook here to add custom service providers via
         * `ServiceProviderCollection::add_service_provider()`.
         *
         * @param Framework\Service\ServiceProvidersCollection $providers
         */
        do_action(ACTION_ADD_SERVICE_PROVIDERS, $providers);

        $bootstrapped = $multilingualpress->bootstrap();

        unset($providers);
    } catch (\Throwable $thr) {
        do_action(ACTION_LOG, 'error', $thr->getMessage(), compact('thr'));

        if (defined('WP_DEBUG') && WP_DEBUG) {
            throw $thr;
        }
    }

    return $bootstrapped;
}

/**
 * Triggers a plugin-specific activation action third parties can listen to.
 *
 * @wp-hook activate_{$plugin}
 */
function activate()
{
    /**
     * Fires when MultilingualPress is about to be activated.
     */
    do_action(ACTION_ACTIVATION);

    add_action(
        'activated_plugin',
        static function (string $plugin) {
            if (plugin_basename(__FILE__) === $plugin) {
                // Bootstrap MultilingualPress to take care of installation or upgrade routines.
                bootstrap();
            }
        },
        0
    );
}

if (version_compare(PHP_VERSION, '7.2', '<')) {
    add_action('network_admin_notices', static function () {
        $message = __(
            'As such, using outdated and unsupported versions of PHP may expose your site to security vulnerabilities.
             Please update PHP to the latest version.
             For further informations see
             <a href="https://multilingualpress.org/docs/php-7-2-update/" target="_blank">here</a>
             and if any questions remain contact our support team.',
            'multilingualpress'
        );
        printf(
            '<div class="notice notice-info"><span class="notice-title">%1$s</span><p>%2$s</p></div>',
            esc_html__(
                'MultilingualPress is dropping support for PHP versions less than 7.2.',
                'multilingualpress'
            ),
            wp_kses($message, ['a' => ['href' => true, 'target' => true]])
        );
    });
}

if (version_compare(get_bloginfo('version'), '5.0', '<')) {
    add_action('network_admin_notices', static function () {
        $message = __(
            'Please update WordPress to the latest version. Without an update, you will not receive any new features.
             Contact our support team <a href="https://multilingualpress.org/support/" target="_blank">here</a>
             if any questions remain.',
            'multilingualpress'
        );
        printf(
            '<div class="notice notice-info"><span class="notice-title">%1$s</span><p>%2$s</p></div>',
            esc_html__(
                'MultilingualPress is dropping support for WordPress versions less than 5.0.',
                'multilingualpress'
            ),
            wp_kses($message, ['a' => ['href' => true, 'target' => true]])
        );
    });
}

(static function (string $rootFile) {
    $rootDir = dirname($rootFile);

    autoload($rootDir);

    $modularity = "{$rootDir}/src/inc/modularity.php";
    if (file_exists($modularity)) {
        $moduleActivator = require_once $modularity;
        $moduleActivator($rootDir);
    }

    add_action('plugins_loaded', __NAMESPACE__ . '\\bootstrap', 0);

    register_activation_hook($rootFile, __NAMESPACE__ . '\\activate');
})(__FILE__);
