<?php # -*- coding: utf-8 -*-
/*
 * This file is part of the MultilingualPress package.
 *
 * (c) Inpsyde GmbH
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\Module\QuickLinks;

use Inpsyde\MultilingualPress\Asset\AssetFactory;
use Inpsyde\MultilingualPress\Core\Admin\PluginSettingsUpdater;
use Inpsyde\MultilingualPress\Core\Admin\SiteSettingsRepository;
use Inpsyde\MultilingualPress\Core\Locations;
use Inpsyde\MultilingualPress\Core\ServiceProvider as CoreServiceProvider;
use Inpsyde\MultilingualPress\Framework\Admin\SettingsPageTab;
use Inpsyde\MultilingualPress\Framework\Admin\SettingsPageTabData;
use Inpsyde\MultilingualPress\Framework\Api\ContentRelations;
use Inpsyde\MultilingualPress\Framework\Asset\AssetManager;
use Inpsyde\MultilingualPress\Framework\Factory\NonceFactory;
use Inpsyde\MultilingualPress\Framework\Module\Module;
use Inpsyde\MultilingualPress\Framework\Module\ModuleManager;
use Inpsyde\MultilingualPress\Framework\Module\ModuleServiceProvider;
use Inpsyde\MultilingualPress\Framework\PluginProperties;
use Inpsyde\MultilingualPress\Framework\Service\Container;
use Inpsyde\MultilingualPress\Module\QuickLinks\Model\CollectionFactory;
use Inpsyde\MultilingualPress\Module\QuickLinks\Settings\QuickLinksPositionViewModel;
use Inpsyde\MultilingualPress\Module\QuickLinks\Settings\Repository;
use Inpsyde\MultilingualPress\Module\QuickLinks\Settings\TabView;
use wpdb;

/**
 * Class ServiceProvider
 * @package Inpsyde\MultilingualPress\Module\QuickLinks
 */
class ServiceProvider implements ModuleServiceProvider
{
    const MODULE_ID = 'quick_links';

    /**
     * @inheritDoc
     *
     * phpcs:disable Inpsyde.CodeQuality.FunctionLength.TooLong
     */
    public function register(Container $container)
    {
        // phpcs:enable

        $container->addService(
            ValidateRedirectFilter::class,
            function (Container $container): ValidateRedirectFilter {
                return new ValidateRedirectFilter(
                    $container[wpdb::class]
                );
            }
        );

        $container->addService(
            Redirector::class,
            function (Container $container): Redirector {
                return new Redirector(
                    $container[NonceFactory::class]->create(['quicklinks_redirector'])
                );
            }
        );

        $container->addService(
            CollectionFactory::class,
            function (Container $container): CollectionFactory {
                return new CollectionFactory(
                    $container[ContentRelations::class],
                    $container[SiteSettingsRepository::class]
                );
            }
        );

        $container->addService(
            QuickLink::class,
            function (Container $container): QuickLink {
                return new QuickLink(
                    $container[CollectionFactory::class],
                    $container[NonceFactory::class]->create(['quicklinks_redirector']),
                    $container[Repository::class]
                );
            }
        );

        /* ----------------------------------------------------
           Settings
           ------------------------------------------------- */

        $container->addService(
            Repository::class,
            function (): Repository {
                return new Repository();
            }
        );

        $container->addService(
            Settings\Updater::class,
            function (Container $container): Settings\Updater {
                return new Settings\Updater(
                    $container[NonceFactory::class]->create(['save_module_quicklinks_settings']),
                    $container[Repository::class]
                );
            }
        );

        $settingsRepository = $container[Repository::class];
        $nonceFactory = $container[NonceFactory::class];
        $moduleManager = $container[ModuleManager::class];

        add_filter(
            CoreServiceProvider::ACTION_BUILD_TABS,
            function (array $tabs) use ($settingsRepository, $nonceFactory, $moduleManager) {

                if ($moduleManager->isModuleActive(self::MODULE_ID)) {
                    $tabs['quicklinks'] = new SettingsPageTab(
                        new SettingsPageTabData(
                            'quicklinks',
                            esc_html_x(
                                'QuickLinks',
                                'QuickLinks Module Settings',
                                'multilingualpress'
                            ),
                            'quicklinks'
                        ),
                        new TabView(
                            $nonceFactory->create(['save_module_quicklinks_settings']),
                            new QuickLinksPositionViewModel($settingsRepository)
                        )
                    );
                }

                return $tabs;
            }
        );
    }

    /**
     * @inheritDoc
     */
    public function registerModule(ModuleManager $moduleManager): bool
    {
        return $moduleManager->register(
            new Module(
                self::MODULE_ID,
                [
                    'description' => __(
                        'Show link to translations in post content.',
                        'multilingualpress'
                    ),
                    'name' => __('QuickLink', 'multilingualpress'),
                    'active' => false,
                ]
            )
        );
    }

    /**
     * @inheritDoc
     */
    public function activateModule(Container $container)
    {
        $pluginProperties = $container[PluginProperties::class];

        $locations = new Locations();
        $locations
            ->add(
                'css',
                $pluginProperties->dirPath() . 'src/modules/QuickLinks/public/css',
                $pluginProperties->dirUrl() . 'src/modules/QuickLinks/public/css'
            )
            ->add(
                'js',
                $pluginProperties->dirPath() . 'src/modules/QuickLinks/public/js',
                $pluginProperties->dirUrl() . 'src/modules/QuickLinks/public/js'
            );

        is_admin()
            ? $this->activateModuleForAdmin($container, $locations)
            : $this->activateModuleForFrontend($container, $locations);
    }

    /**
     * Activate Module for Admin
     *
     * @param Container $container
     * @param Locations $locations
     */
    protected function activateModuleForAdmin(Container $container, Locations $locations)
    {
        $this->setupScriptsForAdmin($container, $locations);

        add_action(
            PluginSettingsUpdater::ACTION_UPDATE_PLUGIN_SETTINGS,
            [$container[Settings\Updater::class], 'updateSettings']
        );
    }

    /**
     * Register and Enqueue Scripts for Admin
     *
     * @param Container $container
     * @param Locations $locations
     */
    protected function setupScriptsForAdmin(Container $container, Locations $locations)
    {
        $assetFactory = new AssetFactory($locations);
        $container[AssetManager::class]
            ->registerStyle(
                $assetFactory->createInternalStyle(
                    'multilingualpress-quicklinks-admin',
                    'admin.min.css',
                    [],
                    null,
                    'screen'
                )
            );

        $container[AssetManager::class]->enqueueStyle('multilingualpress-quicklinks-admin');
    }

    /**
     * Activate module for Frontend
     *
     * @param Container $container
     * @param Locations $locations
     */
    protected function activateModuleForFrontend(Container $container, Locations $locations)
    {
        add_filter('the_content', [$container[QuickLink::class], 'filter']);
        add_action('wp_loaded', [$container[Redirector::class], 'redirect']);

        $this->setupValidateRedirectFilter($container);
        $this->setupScriptsForFrontend($container, $locations);
    }

    /**
     * Register and Enqueue Scripts for Frontend
     *
     * @param Container $container
     * @param Locations $locations
     */
    protected function setupScriptsForFrontend(Container $container, Locations $locations)
    {
        $assetFactory = new AssetFactory($locations);
        $container[AssetManager::class]
            ->registerStyle(
                $assetFactory->createInternalStyle(
                    'multilingualpress-quicklinks-front',
                    'frontend.min.css',
                    [],
                    null,
                    'screen'
                )
            )
            ->registerScript(
                $assetFactory->createInternalScript(
                    'multilingualpress-quicklinks-front',
                    'frontend.min.js'
                )
            );

        $container[AssetManager::class]->enqueueStyle('multilingualpress-quicklinks-front');
        $container[AssetManager::class]->enqueueScript('multilingualpress-quicklinks-front');
    }

    /**
     * Setup ValidateRedirectFilter
     */
    protected function setupValidateRedirectFilter(Container $container)
    {
        $filter = $container[ValidateRedirectFilter::class];
        $filter->enable();
        add_action(Redirector::ACTION_AFTER_VALIDATE_REDIRECT, [$filter, 'disable']);
    }
}
