<?php # -*- coding: utf-8 -*-
/*
 * This file is part of the MultilingualPress package.
 *
 * (c) Inpsyde GmbH
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\SiteDuplication;

use Inpsyde\MultilingualPress\Attachment;
use Inpsyde\MultilingualPress\Core\Admin;
use Inpsyde\MultilingualPress\Framework\Api\ContentRelations;
use Inpsyde\MultilingualPress\Framework\Asset\AssetManager;
use Inpsyde\MultilingualPress\Framework\BasePathAdapter;
use Inpsyde\MultilingualPress\Framework\Factory\NonceFactory;
use Inpsyde\MultilingualPress\Framework\Database\TableDuplicator;
use Inpsyde\MultilingualPress\Framework\Database\TableList;
use Inpsyde\MultilingualPress\Framework\Database\TableReplacer;
use Inpsyde\MultilingualPress\Framework\Database\TableStringReplacer;
use Inpsyde\MultilingualPress\Framework\Filesystem;
use Inpsyde\MultilingualPress\Framework\Http\ServerRequest;
use Inpsyde\MultilingualPress\Framework\Nonce\Nonce;
use Inpsyde\MultilingualPress\Framework\Nonce\SiteAwareNonce;
use Inpsyde\MultilingualPress\Framework\Setting\Site\SiteSettingMultiView;
use Inpsyde\MultilingualPress\Framework\Setting\Site\SiteSettingsSectionView;
use Inpsyde\MultilingualPress\Core\Admin\NewSiteSettings;
use Inpsyde\MultilingualPress\Framework\Service\BootstrappableServiceProvider;
use Inpsyde\MultilingualPress\Framework\Service\Container;
use Inpsyde\MultilingualPress\Schedule\AjaxScheduleHandler;
use Inpsyde\MultilingualPress\Schedule\Scheduler;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\AttachmentDuplicatorHandler;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\AttachmentDuplicatorScheduler;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\NewSiteScheduleTemplate;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\ScheduleAssetManager;
use Inpsyde\MultilingualPress\SiteDuplication\Schedule\ScheduleOption;
use function Inpsyde\MultilingualPress\wpHookProxy;

/**
 * Service provider for all site duplication objects.
 */
final class ServiceProvider implements BootstrappableServiceProvider
{
    /**
     * @inheritdoc
     *
     * phpcs:disable Inpsyde.CodeQuality.FunctionLength.TooLong
     */
    public function register(Container $container)
    {
        // phpcs:enable

        $container->addService(
            ActivePlugins::class,
            function (): ActivePlugins {
                return new ActivePlugins();
            }
        );

        $container->addService(
            Attachment\Duplicator::class,
            function (Container $container): Attachment\Duplicator {
                return new Attachment\Duplicator(
                    $container[BasePathAdapter::class],
                    $container[Filesystem::class]
                );
            }
        );

        $container->addService(
            ActivatePluginsSetting::class,
            function (): ActivatePluginsSetting {
                return new ActivatePluginsSetting();
            }
        );

        $container->addService(
            BasedOnSiteSetting::class,
            function (Container $container): BasedOnSiteSetting {
                return new BasedOnSiteSetting(
                    $container[\wpdb::class],
                    $this->duplicateNonce($container)
                );
            }
        );

        $container->addService(
            SearchEngineVisibilitySetting::class,
            function (): SearchEngineVisibilitySetting {
                return new SearchEngineVisibilitySetting();
            }
        );

        $container->addService(
            SiteDuplicator::class,
            function (Container $container): SiteDuplicator {
                return new SiteDuplicator(
                    $container[\wpdb::class],
                    $container[TableList::class],
                    $container[TableDuplicator::class],
                    $container[TableReplacer::class],
                    $container[ActivePlugins::class],
                    $container[ContentRelations::class],
                    $container[ServerRequest::class],
                    $this->duplicateNonce($container)
                );
            }
        );

        $container->addService(
            ScheduleOption::class,
            function (): ScheduleOption {
                return new ScheduleOption();
            }
        );

        $container->addService(
            AttachmentDuplicatorScheduler::class,
            function (Container $container): AttachmentDuplicatorScheduler {
                return new AttachmentDuplicatorScheduler(
                    $container[ScheduleOption::class],
                    $container[Attachment\Collection::class],
                    $container[Scheduler::class]
                );
            }
        );

        $container->addService(
            Attachment\DatabaseDataReplacer::class,
            function (Container $container): Attachment\DatabaseDataReplacer {
                return new Attachment\DatabaseDataReplacer(
                    $container[\wpdb::class],
                    $container[TableStringReplacer::class],
                    $container[BasePathAdapter::class]
                );
            }
        );

        $container->addService(
            AttachmentDuplicatorHandler::class,
            function (Container $container): AttachmentDuplicatorHandler {
                return new AttachmentDuplicatorHandler(
                    $container[ScheduleOption::class],
                    $container[Attachment\Duplicator::class],
                    $container[Attachment\Collection::class],
                    $container[Scheduler::class],
                    $container[Attachment\DatabaseDataReplacer::class]
                );
            }
        );

        $container->addService(
            ScheduleAssetManager::class,
            function (Container $container): ScheduleAssetManager {
                return new ScheduleAssetManager(
                    $container[ScheduleOption::class],
                    $container[AjaxScheduleHandler::class],
                    $container[AssetManager::class]
                );
            }
        );
    }

    /**
     * @inheritdoc
     */
    public function bootstrap(Container $container)
    {
        $siteDuplicator = $container[SiteDuplicator::class];
        $siteSettingsUpdater = $container[Admin\SiteSettingsUpdater::class];
        $attachmentDuplicatorScheduler = $container[AttachmentDuplicatorScheduler::class];
        $attachmentDuplicatorHandler = $container[AttachmentDuplicatorHandler::class];
        $settingView = SiteSettingMultiView::fromViewModels(
            [
                $container[BasedOnSiteSetting::class],
                $container[ActivatePluginsSetting::class],
                $container[SearchEngineVisibilitySetting::class],
            ]
        );

        add_action(
            SiteSettingsSectionView::ACTION_AFTER . '_' . NewSiteSettings::SECTION_ID,
            [$settingView, 'render']
        );

        add_action(
            'network_site_new_form',
            [$container[ScheduleAssetManager::class], 'enqueueLocalizedScript']
        );
        add_action('admin_footer', [new NewSiteScheduleTemplate(), 'render']);

        add_action('wpmu_new_blog', wpHookProxy([$siteDuplicator, 'duplicateSite']), 0);
        add_action('wpmu_new_blog', wpHookProxy([$siteSettingsUpdater, 'defineInitialSettings']));

        add_action(
            SiteDuplicator::DUPLICATE_ACTION_KEY,
            [$attachmentDuplicatorScheduler, 'schedule'],
            10,
            2
        );
        add_action(
            AttachmentDuplicatorScheduler::SCHEDULE_HOOK,
            [$attachmentDuplicatorHandler, 'handle']
        );
    }

    /**
     * @param Container $container
     * @return Nonce
     */
    private function duplicateNonce(Container $container): Nonce
    {
        $nonce = $container[NonceFactory::class]->create(['duplicate_site']);
        // When creating a new site, its ID is not yet known, so create a nonce for a fixed site ID 0.
        if ($nonce instanceof SiteAwareNonce) {
            $nonce = $nonce->withSite(0);
        }

        return $nonce;
    }
}
