<?php # -*- coding: utf-8 -*-
/*
 * This file is part of the MultilingualPress package.
 *
 * (c) Inpsyde GmbH
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace Inpsyde\MultilingualPress\SiteDuplication\Schedule;

use Inpsyde\MultilingualPress\Framework\Asset\AssetManager;
use Inpsyde\MultilingualPress\Schedule\AjaxScheduleHandler;

/**
 * Class ScheduleAssetManager
 * @package Inpsyde\MultilingualPress\SiteDuplication
 */
class ScheduleAssetManager
{
    /**
     * @var ScheduleOption
     */
    private $scheduleOption;

    /**
     * @var AjaxScheduleHandler
     */
    private $ajaxScheduleHandler;

    /**
     * @var AssetManager
     */
    private $assetManager;

    /**
     * ScheduleAssetManager constructor.
     * @param ScheduleOption $scheduleOption
     * @param AjaxScheduleHandler $ajaxScheduleHandler
     * @param AssetManager $assetManager
     */
    public function __construct(
        ScheduleOption $scheduleOption,
        AjaxScheduleHandler $ajaxScheduleHandler,
        AssetManager $assetManager
    ) {

        $this->scheduleOption = $scheduleOption;
        $this->ajaxScheduleHandler = $ajaxScheduleHandler;
        $this->assetManager = $assetManager;
    }

    /**
     * Enqueue and Localize the main plugin script
     *
     * @wp-hook network_site_new_form
     *
     * @return void
     */
    public function enqueueLocalizedScript()
    {
        $scheduleAjaxUrls = $this->scheduleAjaxInfoUrls();

        $this->assetManager->enqueueScriptWithData(
            'multilingualpress-admin',
            'siteDuplicatorScheduleData',
            [
                'scheduleInfoUrl' => $scheduleAjaxUrls,
                'translations' => [
                    'doNotCloseBrowserTabMessage' => esc_html__(
                        'You cannot close this window until the entire process is finished.',
                        'multilingualpress'
                    ),
                    'scheduleInfoErrorMessage' => esc_html__(
                        'Something went wrong when trying to retrieve information about media copy task. Please, wait a minute or try to reload the page. If the problem persists please contact our support.',
                        'multilingualpress'
                    ),
                    'preventFormSubmissionMessage' => esc_html__(
                        'Actually one or more scheduled jobs are running, please wait until everything has been completed.',
                        'multilingualpress'
                    ),
                ],
            ]
        );
    }

    /**
     * Retrieve the ajax schedule information url to call to obtain information about the current
     * status of the cron jobs
     *
     * @return array
     */
    private function scheduleAjaxInfoUrls(): array
    {
        $scheduleAjaxUrls = [];
        foreach ($this->scheduleOption->allSchedule() as $siteId => $scheduleId) {
            $scheduleAjaxUrls[$siteId] = $this->ajaxScheduleHandler->scheduleInfoAjaxUrl($scheduleId);
        }

        return $scheduleAjaxUrls;
    }
}
