<?php

class GF_LaPosta_API
{

	/**
	 * @param mixed $api_url 
	 * @param mixed $api_key 
	 * @return void 
	 */
	function __construct($api_url, $api_key = null)
	{

		$this->api_url = $api_url;
		$this->api_key = $api_key;
	}

	/** @return array  */
	function default_options()
	{

		return array(
			'api_key'    => $this->api_key,
			'api_output' => 'json',
		);
	}

	/**
	 * @param mixed $action 
	 * @param array $options 
	 * @param string $method 
	 * @return mixed 
	 */
	function make_request($action, $options = array(), $method = 'GET')
	{
		/* Build request options string. */
		$request_options               = $this->default_options();
		$request_options['api_action'] = $action;
		$request_task = $action;
		$client_id  = $this->api_key;
		$client_pass = ':';

		$args = array(
			'headers'     => array(
				'Authorization'    => 'Basic ' . base64_encode("$client_id:$client_pass"),
			),
		);

		$request_options = http_build_query($request_options);
		$request_options .= ($method == 'GET') ? '&' . http_build_query($options) : null;

		if($request_task == 'member_update' && array_key_exists('member_id', $options)){
			$request_task = 'member/' .$options['member_id'];
		}

		/* Build request URL. */
		$request_url = untrailingslashit($this->api_url) . '/v2/' . $request_task;

	
		/**
		 * Allows request timeout to to be changed. Timeout is in seconds
		 *
		 * @since 1.5
		 */
		$timeout = apply_filters('gform_laposta_request_timeout', 30);

		/* Execute request based on method. */
		switch ($method) {

			case 'POST':

				$args_post     = array(
					'headers'     => array(
						'Authorization'    => 'Basic ' . base64_encode("$client_id:$client_pass"),
					),
					'body' =>  $options,
					'timeout' => $timeout,
				);
				$response = wp_remote_post($request_url, $args_post);
				break;

			case 'GET':
				$response = wp_remote_get($request_url, $args);
				break;
		}
		$response_code  = wp_remote_retrieve_response_code($response);

		/* If WP_Error, die. Otherwise, return decoded JSON. */
		if (is_wp_error($response) || ($response_code !== 200 && $response_code !== 201)) {
			//die('Request failed. ' . $response->get_error_message());
			if ($response_code == 400 && isset($response)) {
				$body = wp_remote_retrieve_body($response);
				$data_object = json_decode($body);
				if ($data_object && property_exists($data_object, 'error')) {
					return $data_object->error;
				}
			}
		} else {
			$body = wp_remote_retrieve_body($response);

			$data_object = json_decode($body);

			if ($data_object && property_exists($data_object, 'data')) {
				return $data_object->data;
			}
			if ($data_object && property_exists($data_object, 'list')) {
				return $data_object->list;
			}
			if ($data_object && property_exists($data_object, 'member')) {
				return $data_object->member;
			}
			if ($data_object && property_exists($data_object, 'error')) {
				return $data_object->error;
			}
			//	die('Conversion failed. ');
		}
	}

	/**
	 * Test the provided API credentials.
	 *
	 * @access public
	 * @return bool
	 */
	function auth_test()
	{
		/* Build options string. */
		$request_options               = $this->default_options();
		$request_options['api_action'] = 'list_paginator';
		$request_options               = http_build_query($request_options);
		//Credentials
		$client_id  = $this->api_key;
		$client_pass = ':';
		$args = array(
			'headers'     => array(
				'Authorization'    => 'Basic ' . base64_encode("$client_id:$client_pass"),
			)
		);
		//	'httpsDisableVerifyPeer' => $use_https

		/* Setup request URL. */
		$request_url = untrailingslashit($this->api_url) . '/v2/list';

		/* Execute request. */
		$response = wp_remote_get($request_url, $args);
		$response_code  = wp_remote_retrieve_response_code($response);

		/* If invalid content type, API URL is invalid. */
		if (is_wp_error($response) || $response_code !== 200) {
			throw new Exception('Invalid API URL.');
		} else {
			/* If result code is false, API key is invalid. */
			if ($response_code == 401) {
				throw new Exception('Unauthorized');
			}
		}

		return true;
	}


	/**
	 * Add a new custom list field.
	 *
	 * @access public
	 *
	 * @param array $custom_field
	 *
	 * @return array
	 */
	function add_custom_field($custom_field, $list_id)
	{
		$action = array('id' => $list_id, 'name' => $custom_field);
		return $this->make_request('field', $action , 'POST');
	}

	/**
	 * Get all custom list fields.
	 *
	 * @access public
	 * @return array
	 */
	function get_custom_fields($list_id)
	{
		if (!empty($list_id)) {
			$request_task = 'field' . '?list_id=' . $list_id;
			return $this->make_request($request_task, array('id' => $list_id));
		}
	}

	/**
	 * Get all forms in the system.
	 *
	 * @access public
	 * @return array
	 */
	function get_forms()
	{

		//return $this->make_request( 'form_getforms' );
		return null;
	}

	/**
	 * Get specific list.
	 *
	 * @access public
	 *
	 * @param int $list_id
	 *
	 * @return array
	 */
	function get_list($list_id)
	{
		if (!empty($list_id)) {
			$request_task = 'list/' . $list_id;
			return $this->make_request($request_task, array('id' => $list_id));
		}
		return $this->make_request('list', array('id' => $list_id));
	}

	/**
	 * Get all lists in the system.
	 *
	 * @access public
	 * @return array
	 */
	function get_lists()
	{

		return $this->make_request('list', array('ids' => 'all'));
	}

	/**
	 * Add or update a contact/member
	 *
	 * @access public
	 *
	 * @param mixed $contact
	 *
	 * @return array
	 */
	function sync_contact($contact)
	{
		$result = $this->make_request('member', $contact, 'POST');

		// existing member do update
		if(property_exists( $result, 'member_id') && property_exists( $result, 'code')){
			if($result->code == 204){
				$contact['member_id'] = $result->member_id;
				$result = $this->update_contact($contact);
			}
		}
		return $result;
	}

	/**
	 * update a contact/member
	 * @param mixed $contact 
	 * @return mixed 
	 */
	function update_contact($contact)
	{
		$result = $this->make_request('member_update', $contact, 'POST');
		return $result;
	}


	/**
	 * Add note to contact.
	 */
	function add_note($contact_id, $list_id, $note)
	{

		$request = array(
			'id'     => $contact_id,
			'listid' => $list_id,
			'note'   => $note
		);

		return $this->make_request('contact_note_add', $request, 'POST');
	}
}
