<?php

// don't load directly
if ( ! defined( 'ABSPATH' ) ) {
	die();
}

GFForms::include_feed_addon_framework();

/**
 * Gravity Forms LaPosta Add-On.
 *
 * @since     1.0
 * @package   GravityForms
 * @author    Rocketgenius
 * @copyright Copyright (c) 2016, Rocketgenius
 */
class GFLaPosta extends GFFeedAddOn {

	/**
	 * Contains an instance of this class, if available.
	 *
	 * @since  1.0
	 * @access private
	 * @var    object $_instance If available, contains an instance of this class.
	 */
	private static $_instance = null;

	/**
	 * Defines the version of the LaPosta Add-On.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_version Contains the version, defined from laposta.php
	 */
	protected $_version = GF_LAPOSTA_VERSION;

	/**
	 * Defines the minimum Gravity Forms version required.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_min_gravityforms_version The minimum version required.
	 */
	protected $_min_gravityforms_version = '1.9.14.26';

	/**
	 * Defines the plugin slug.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_slug The slug used for this plugin.
	 */
	protected $_slug = 'gravityformslaposta';

	/**
	 * Defines the main plugin file.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_path The path to the main plugin file, relative to the plugins folder.
	 */
	protected $_path = 'gravityformslaposta/laposta.php';

	/**
	 * Defines the full path to this class file.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_full_path The full path.
	 */
	protected $_full_path = __FILE__;

	/**
	 * Defines the URL where this Add-On can be found.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string The URL of the Add-On.
	 */
	protected $_url = 'http://www.gravityforms.com';

	/**
	 * Defines the title of this Add-On.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_title The title of the Add-On.
	 */
	protected $_title = 'Gravity Forms LaPosta Add-On';

	/**
	 * Defines the short title of the Add-On.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_short_title The short title.
	 */
	protected $_short_title = 'LaPosta';

	/**
	 * Defines if Add-On should use Gravity Forms servers for update data.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    bool
	 */
	protected $_enable_rg_autoupgrade = true;
	/**
	 * Defines the capability needed to access the Add-On settings page.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_capabilities_settings_page The capability needed to access the Add-On settings page.
	 */
	protected $_capabilities_settings_page = 'gravityforms_laposta';

	/**
	 * Defines the capability needed to access the Add-On form settings page.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_capabilities_form_settings The capability needed to access the Add-On form settings page.
	 */
	protected $_capabilities_form_settings = 'gravityforms_laposta';

	/**
	 * Defines the capability needed to uninstall the Add-On.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    string $_capabilities_uninstall The capability needed to uninstall the Add-On.
	 */
	protected $_capabilities_uninstall = 'gravityforms_laposta_uninstall';

	/**
	 * Defines the capabilities needed for the Post Creation Add-On
	 *
	 * @since  1.0
	 * @access protected
	 * @var    array $_capabilities The capabilities needed for the Add-On
	 */
	protected $_capabilities = array( 'gravityforms_laposta', 'gravityforms_laposta_uninstall' );

	/**
	 * Stores an instance of the LaPosta API library, if initialized.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    object $api If initialized, an instance of the LaPosta API library.
	 */
	protected $api = null;

	/**
	 * New LaPosta fields that need to be created when saving feed.
	 *
	 * @since  1.0
	 * @access protected
	 * @var    object $api When saving feed, new LaPosta fields that need to be created.
	 */
	protected $_new_custom_fields = array();

	/**
	 * Get an instance of this class.
	 *
	 * @return GFLaPosta
	 */
	public static function get_instance() {

		if ( null === self::$_instance ) {
			self::$_instance = new GFLaPosta();
		}

		return self::$_instance;

	}


	// # FEED PROCESSING -----------------------------------------------------------------------------------------------


	/**
	 * Process the feed, subscribe the user to the list.
	 *
	 * @param array $feed The feed object to be processed.
	 * @param array $entry The entry object currently being processed.
	 * @param array $form The form object currently being processed.
	 *
	 * @return bool|void
	 */
	public function process_feed( $feed, $entry, $form ) {

		$this->log_debug( __METHOD__ . '(): Processing feed.' );

		/* If API instance is not initialized, exit. */
		if ( ! $this->initialize_api() ) {

			$this->log_error( __METHOD__ . '(): Failed to set up the API.' );

			return;

		}

		/* Setup mapped fields array. */
		$mapped_fields = $this->get_field_map_fields( $feed, 'fields' );

		/* Setup contact array. */
		$email = $this->get_field_value( $form, $entry, rgar( $mapped_fields, 'email' ));
		$ip = $this->get_field_value( $form, $entry, rgar( $mapped_fields, 'ip' ) );

		// IP verplicht, niet te zien in dashboard laposta
		if(empty($ip)){
			$ip = '198.51.100.100';
		}
		$source_url = $this->get_field_value( $form, $entry, rgar( $mapped_fields, 'source_url' ) );

		$contact = array(
			'email'       => 	$email,
			'ip' 		  =>  $ip,
			'source_url'  => $source_url  
		);

		/* If the email address is invalid or empty, exit. */
		if ( GFCommon::is_invalid_or_empty_email( $contact['email'] ) ) {
			$this->add_feed_error( esc_html__( 'A valid Email address must be provided.', 'gravityformslaposta' ), $feed, $entry, $form );

			return $entry;
		}

		/**
		 * Prevent empty form fields from erasing values already stored 
		 * when updating an existing subscriber.
		 *
		 *
		 * @param bool  $override If blank fields should override values already stored
		 * @param array $form     The form object.
		 * @param array $entry    The entry object.
		 * @param array $feed     The feed object.
		 */
		$override_empty_fields = gf_apply_filters( 'gform_laposta_override_empty_fields', array( $form['id'] ), true, $form, $entry, $feed );

		$custom_contactfields  =  array();
		/* Assiging properties that have been mapped */
		$properties = array('voornaam', 'achternaam' );
		foreach ( $properties as $property ) {
			$field_value = $this->get_field_value( $form, $entry, rgar( $mapped_fields, $property ) );
			$is_mapped = ! rgempty( $property, $mapped_fields );

			/* Only send fields that are mapped. Also, make sure blank values are ok to override existing data */
			if ( $is_mapped && ( $override_empty_fields || ! empty( $field_value ) ) ) {
				$custom_contactfields[ $property ] = $field_value;
			}
		}

		/* Prepare tags. */
		if ( rgars( $feed, 'meta/tags' ) ) {

			$tags            = GFCommon::replace_variables( $feed['meta']['tags'], $form, $entry, false, false, false, 'text' );
			$tags            = array_map( 'trim', explode( ',', $tags ) );
			$contact['tags'] = gf_apply_filters( 'gform_laposta_tags', $form['id'], $tags, $feed, $entry, $form );

		}

		/* Add list to contact array. */
		$list_id = $feed['meta']['list'];
		$contact[ 'list_id'] =  $list_id;
	
		/* Add custom fields to contact array. */
		$custom_fields = rgars( $feed, 'meta/custom_fields' );
		if ( is_array( $custom_fields ) ) {
			foreach ( $feed['meta']['custom_fields'] as $custom_field ) {

				if ( rgblank( $custom_field['key'] ) || 'gf_custom' === $custom_field['key'] || rgblank( $custom_field['value'] ) ) {
					continue;
				}

				$field_value = $this->get_field_value( $form, $entry, $custom_field['value'] );

				if ( rgblank( $field_value ) && ! $override_empty_fields ) {
					continue;
				}

				$contact_key = $custom_field['key'];

				//If contact is already set, don't override it with fields that are hidden by conditional logic
				$is_hidden = GFFormsModel::is_field_hidden( $form, GFFormsModel::get_field( $form, $custom_field['value'] ), array(), $entry );
				if ( isset( $custom_contactfields[ $contact_key ] ) && $is_hidden ) {
					continue;
				}
/* 				if (strpos($field_value, ',') !== false) {
					$field_values = explode(',', $field_value);
					foreach ($field_values as $value) {
						$custom_contactfields[$contact_key][] = trim($value);
					}
				} else {
					$custom_contactfields[$contact_key] = $field_value;
				} */

				$custom_contactfields[$contact_key] = $field_value;
			}
		}

		/* add */
		$contact['custom_fields'] = $custom_contactfields;

		// options
		$options_contactfields  =  array();
		$options_contactfields['ignore_doubleoptin'] = true;
		$options_contactfields['suppress_email_notification'] = true;	
		$options_contactfields['suppress_email_welcome'] = true;	
		$contact['options'] =  	$options_contactfields;

		/**
		 * Allows the contact properties to be overridden before the contact_sync request is sent to the API.
		 *
		 * @param array $contact The contact properties.
		 * @param array $entry The entry currently being processed.
		 * @param array $form The form object the current entry was created from.
		 * @param array $feed The feed which is currently being processed.
		 *
		 * @since 1.3.5
		 */
		$contact = apply_filters( 'gform_laposta_contact_pre_sync', $contact, $entry, $form, $feed );
		$contact = apply_filters( 'gform_laposta_contact_pre_sync_' . $form['id'], $contact, $entry, $form, $feed );

		/* Sync the contact. */
		$this->log_debug( __METHOD__ . '(): Contact to be added => ' . print_r( $contact, true ) );
		//  $list_id = Lapost listid
		$sync_contact = $this->api->sync_contact( $contact, $list_id);
		
		//todo
		if ( $sync_contact ) {
			
			if(property_exists( $sync_contact, 'member_id')){

				$this->log_debug( __METHOD__ . '(): Information send to Laposta feed.' );

				$note_message =  esc_html__( 'The information was received by Laposta', 'gravityformslaposta' );
				$this->add_note( $entry['id'], $note_message, 'success');
			}
		} 
		else {

			$this->log_error( __METHOD__ . "(): {$contact['email']} was not added; {$sync_contact['result_message']}" );

			return false;

		}
		return true;
	}

	// # ADMIN FUNCTIONS -----------------------------------------------------------------------------------------------

	/**
	 * Plugin starting point. Handles hooks, loading of language files and PayPal delayed payment support.
	 */
	public function init() {

		parent::init();

		$this->add_delayed_payment_support(
			array(
				'option_label' => esc_html__( 'Subscribe contact to LaPosta only when payment is received.', 'gravityformslaposta' )
			)
		);

	}

	/**
	 * Return the stylesheets which should be enqueued.
	 *
	 * @return array
	 */
	public function styles() {

		$min    = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG || isset( $_GET['gform_debug'] ) ? '' : '.min';
		$min = '';
		$styles = array(
			array(
				'handle'  => 'gform_laposta_form_settings_css',
				'src'     => $this->get_base_url() . "/css/form_settings{$min}.css",
				'version' => $this->_version,
				'enqueue' => array(
					array( 'admin_page' => array( 'form_settings' ) ),
				),
			),
		);

		return array_merge( parent::styles(), $styles );

	}

		/**
	 * Enqueue admin scripts.
	 *
	 * @since 1.0
	 *
	 * @return array
	 */
	public function scripts() {

		$min     = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG || isset( $_GET['gform_debug'] ) ? '' : '.min';
		$min = '';
		$form_id = absint( rgget( 'id' ) );
		$form    = GFAPI::get_form( $form_id );
		$scripts = array(
			array(
				'handle'  => 'gform_laposta_pluginsettings',
				'deps'    => array( 'jquery' ),
				'src'     => $this->get_base_url() . "/js/plugin_settings{$min}.js",
				'version' => $this->_version,
				'enqueue' => array(
					array(
						'admin_page' => array( 'plugin_settings' ),
						'tab'        => $this->_slug,
					),
				)
			),
		);

		return array_merge( parent::scripts(), $scripts );

	}

	/**
	 * Return the plugin's icon for the plugin/form settings menu.
	 *
	 * @since 1.8
	 *
	 * @return string
	 */
	public function get_menu_icon() {

		return file_get_contents( $this->get_base_path() . '/images/menu-icon.svg' );

	}

	// # PLUGIN SETTINGS -----------------------------------------------------------------------------------------------

	/**
	 * Configures the settings which should be rendered on the add-on settings tab.
	 *
	 * @return array
	 */
	public function plugin_settings_fields()
	{

		return array(
			array(
				'title'       => '',
				'description' => $this->plugin_settings_description(),
				'fields'      => array(
					array(
						'name'              => 'api_url',
						'label'             => esc_html__('API URL', 'gravityformslaposta'),
						'type'              => 'text',
						'class'             => 'medium',
						'feedback_callback' => array($this, 'initialize_api')
					),
					array(
						'name'              => 'api_key',
						'label'             => esc_html__('API Key', 'gravityformslaposta'),
						'type'              => 'text',
						'class'             => 'large',
						'feedback_callback' => array($this, 'initialize_api')
					),
					array(
						'type'     => 'save',
						'messages' => array(
							'success' => esc_html__('LaPosta settings have been updated.', 'gravityformslaposta')
						),
					),
				),
			),
		);
	}

	/**
	 * Prepare plugin settings description.
	 *
	 * @return string
	 */
	public function plugin_settings_description() {

		$description = '<p>';
		$description .= sprintf(
			esc_html__( 'LaPosta makes it easy to send email newsletters to your customers, manage your subscriber lists, and track campaign performance. Use Gravity Forms to collect customer information and automatically add it to your LaPosta list. If you don\'t have an LaPosta account, you can %1$ssign up for one here.%2$s', 'gravityformslaposta' ),
			'<a href="http://www.laposta.nl/" target="_blank">', '</a>'
		);
		$description .= '</p>';

		if ( ! $this->initialize_api() ) {

			$description .= '<p>';
			$description .= esc_html__( 'Gravity Forms LaPosta Add-On requires your API URL and API Key, which can be found in the API tab on the account settings page.', 'gravityformslaposta' );
			$description .= '</p>';

		}

		return $description;

	}

	// ------- Feed page -------

	/**
	 * Prevent feeds being listed or created if the api key isn't valid.
	 *
	 * @return bool
	 */
	public function can_create_feed() {

		return $this->initialize_api();

	}

	/**
	 * Enable feed duplication.
	 *
	 * @access public
	 * @return bool
	 */
	public function can_duplicate_feed( $id ) {

		return true;

	}

	/**
	 * If the api keys are invalid or empty return the appropriate message.
	 *
	 * @return string
	 */
	public function configure_addon_message() {

		$settings_label = sprintf( esc_html__( '%s Settings', 'gravityforms' ), $this->get_short_title() );
		$settings_link  = sprintf( '<a href="%s">%s</a>', esc_url( $this->get_plugin_settings_url() ), $settings_label );

		if ( is_null( $this->initialize_api() ) ) {

			return sprintf( esc_html__( 'To get started, please configure your %s.', 'gravityforms' ), $settings_link );
		}

		return sprintf( esc_html__( 'Please make sure you have entered valid API credentials on the %s page.', 'gravityformslaposta' ), $settings_link );

	}

	/**
	 * Configures which columns should be displayed on the feed list page.
	 *
	 * @return array
	 */
	public function feed_list_columns() {

		return array(
			'feed_name' => esc_html__( 'Name', 'gravityformslaposta' ),
			'account'      => esc_html__( 'LaPosta Account', 'gravityformslaposta' ),
			'list'      => esc_html__( 'LaPosta List', 'gravityformslaposta' )	
		);

	}

	/**
	 * Returns the value to be displayed in the LaPosta List column.
	 *
	 * @param array $feed The feed being included in the feed list.
	 *
	 * @return string
	 */
	public function get_column_value_list( $feed ) {
		
		/* If LaPosta instance is not initialized, return campaign ID. */
		if ( ! $this->initialize_api() ) {
			return $feed['meta']['list'];
		}

		/* Get campaign and return name */
		$list = $this->api->get_list( $feed['meta']['list'] );
		return (property_exists($list, 'name') ) ?  $list->name : $feed['meta']['list'];
	}

	public function get_column_value_account( $feed ) {
		$result = '';
		/* If LaPosta instance is not initialized, return  */
		if ( ! $this->initialize_api() ) {
			$result = $feed['meta']['account'];
		}
		$result =  (!empty($feed['meta']['account'])) ?  'Custom Key' : '';
		return $result;
	}



	/**
	 * Configures the settings which should be rendered on the feed edit page.
	 *
	 * @return array The feed settings.
	 */
	public function feed_settings_fields() {

		/* Build fields array. */
		$fields = array(
			array(
				'name'          => 'feed_name',
				'label'         => esc_html__( 'Feed Name', 'gravityformslaposta' ),
				'type'          => 'text',
				'required'      => true,
				'default_value' => $this->get_default_feed_name(),
				'class'         => 'medium',
				'tooltip'       => '<h6>' . esc_html__( 'Name', 'gravityformslaposta' ) . '</h6>' . esc_html__( 'Enter a feed name to uniquely identify this setup.', 'gravityformslaposta' ),
			),
			array(
				'name'     => 'list',
				'label'    => esc_html__( 'LaPosta List', 'gravityformslaposta' ),
				'type'     => 'select',
				'required' => true,
				'choices'  => $this->lists_for_feed_setting(),
				'onchange' => "jQuery(this).parents('form').submit();",
				'tooltip'  => '<h6>' . esc_html__( 'LaPosta List', 'gravityformslaposta' ) . '</h6>' . esc_html__( 'Select which LaPosta list this feed will add contacts to.', 'gravityformslaposta' )
			),
			array(
				'name'       => 'fields',
				'label'      => esc_html__( 'Map Fields', 'gravityformslaposta' ),
				'type'       => 'field_map',
				'dependency' => 'list',
				'field_map'  => $this->fields_for_feed_mapping(),
				'tooltip'    => '<h6>' . esc_html__( 'Map Fields', 'gravityformslaposta' ) . '</h6>' . esc_html__( 'Select which Gravity Form fields pair with their respective LaPosta fields.', 'gravityformslaposta' )
			),
			array(
				'name'          => 'custom_fields',
				'label'         =>  esc_html__( 'Map Dynamic Fields', 'gravityformslaposta' ),
				'type'          => 'dynamic_field_map',
				'dependency'    => 'list',
				'field_map'     => $this->custom_fields_for_feed_setting(),
				'save_callback' => array( $this, 'create_new_custom_fields' ),
			)
		);

		/* Add feed condition and options fields. */
		$fields[] = array(
			'name'       => 'feed_condition',
			'label'      => esc_html__( 'Conditional Logic', 'gravityformslaposta' ),
			'type'       => 'feed_condition',
			'dependency' => 'list',
			'tooltip'    => '<h6>' . esc_html__( 'Conditional Logic', 'gravityformslaposta' ) . '</h6>' . esc_html__( 'When conditional logic is enabled, form submissions will only be exported to LaPosta when the condition is met. When disabled, all form submissions will be exported.', 'gravityformslaposta' )
		);
	

		return array(
			array(
				'title'  => 'LaPosta Feed Settings',
				'fields' => $fields
			)
		);

	}

	/**
	 * Renders and initializes a dynamic field map field based on the $field array whose choices are populated by the fields to be mapped.
	 * (Forked to force reload of field map options.)
	 *
	 * @since  Unknown
	 *
	 * @param array $field Field array containing the configuration options of this field.
	 * @param bool  $echo  Determines if field contents should automatically be displayed. Defaults to true.
	 *
	 * @return string
	 */
	public function settings_dynamic_field_map( $field, $echo = true ) {

		// Refresh field map.
		if ( 'custom_fields' === $field['name'] && $this->is_postback() ) {
			$field['field_map'] = $this->custom_fields_for_feed_setting();
		}
		$test = $field['field_map'];
		return parent::settings_dynamic_field_map( $field, $echo );

	}


	/**
	 * Prepare LaPosta accounts lists for feed field
	 *
	 * @return array
	 */
	public function lists_for_accounts_setting() {


		$lists = array(
			array(
				'label' => esc_html__( 'Standaard Account', 'gravityformslaposta' ),
				'value' => ''
			)
		);

		if ( ! $this->initialize_api() ) {
			return $lists;
		}

		$ac_lists = $this->api->get_lists();

		foreach ( $ac_lists as $list ) {
			$lists[] = array(
				'label' => $list->list->name,
				'value' => $list->list->list_id
			);

		}
		if(empty($list)){
			$list = array(' ');
		}

		return $lists;

	}

	/**
	 * Prepare LaPosta lists for feed field
	 *
	 * @return array
	 */
	public function lists_for_feed_setting() {

		$lists = array(
			array(
				'label' => esc_html__( 'Select a List', 'gravityformslaposta' ),
				'value' => ''
			)
		);

		/* If LaPosta API credentials are invalid, return the lists array. */
		if ( ! $this->initialize_api() ) {
			return $lists;
		}

		/* Get available LaPosta lists. */
		$ac_lists = $this->api->get_lists();

		/* Add LaPosta lists to array and return it. */
		foreach ( $ac_lists as $list ) {
			$lists[] = array(
				'label' => $list->list->name,
				'value' => $list->list->list_id
			);

		}

		return $lists;

	}

	/**
	 * Prepare fields for feed field mapping.
	 *
	 * @return array
	 */
	public function fields_for_feed_mapping() {

		$email_field = array(
			'name'       => 'email',
			'label'      => esc_html__( 'Email Address', 'gravityformslaposta' ),
			'required'   => true,
			'tooltip'  => esc_html__( 'Email Field Mapping, required field', 'gravityformslaposta' )
		);

		/**
		 * Allows the list of supported fields types for the email field map to be changed.
		 * Return an array of field types or true (to allow all field types)
		 *
		 * @since 1.5
		 *
		 * @param array|bool $field_types Array of field types or "true" for all field types.
		 */
		$field_types = apply_filters( 'gform_laposta_supported_field_types_email_map', array( 'email', 'ip', 'hidden' ) );

		if ( $field_types !== true & is_array( $field_types ) ) {
			$email_field['field_type'] = $field_types;
		}

		return array(
			$email_field,
			array(
				'name'     => 'voornaam',
				'label'    => esc_html__( 'First Name', 'gravityformslaposta' ),
				'required' => false
			),
			array(
				'name'     => 'achternaam',
				'label'    => esc_html__( 'Last Name', 'gravityformslaposta' ),
				'required' => false
			),
			array(
				'name'     => 'ip',
				'label'    => esc_html__( 'IP address', 'gravityformslaposta' ),
				'tooltip'  => esc_html__( 'Think of the General Data Protection Regulation, field is not always visible everywhere in LaPosta', 'gravityformslaposta' ),
				'required' => false
			),
			array(
				'name'     => 'source_url',
				'label'    => esc_html__( 'Source Url', 'gravityformslaposta' ),
				'required' => false
			)
		);

	}

	/**
	 * Prepare custom fields for feed field mapping.
	 *
	 * @return array
	 */
	public function custom_fields_for_feed_setting() {

		$fields = array();

		/* If LaPosta API credentials are invalid, return the fields array. */
		if ( ! $this->initialize_api() ) {
			return $fields;
		}
		$list_id = $this->get_setting( 'list' );
		/* Get available LaPosta fields. */
		$ac_fields = $this->api->get_custom_fields(	$list_id);

		/* If no LaPosta fields exist, return the fields array. */
		if ( empty( $ac_fields ) ) {
			return $fields;
		}

		/* If LaPosta fields exist, add them to the fields array. */
		foreach($ac_fields as $field=>$value) {
			$field_data = $value->field;

			$fields[] = array(
				'label' => $field_data->name,
				'value' => $field_data->custom_name
			);
		}

		if ( ! empty( $this->_new_custom_fields ) ) {

			foreach ( $this->_new_custom_fields as $new_field ) {

				$found_custom_field = false;
				foreach ( $fields as $field ) {

					if ( $field['value'] == $new_field['value'] ) {
						$found_custom_field = true;
					}

				}

				if ( ! $found_custom_field ) {
					$fields[] = array(
						'label' => $new_field['label'],
						'value' => $new_field['value']
					);
				}

			}

		}


		if ( empty( $fields ) ) {
			return $fields;
		}

		// Add standard "Select a Custom Field" option.
		$standard_field = array(
			array(
				'label' => esc_html__( 'Select a Custom Field', 'gravityformslaposta' ),
				'value' => '',
			),
		);
		$fields = array_merge( $standard_field, $fields );

		/* Add "Add Custom Field" to array. */
		$fields[] = array(
			'label' => esc_html__( 'Add Custom Field', 'gravityformslaposta' ),
			'value' => 'gf_custom',
		);

		return $fields;

	}

	/**
	 * Prepare LaPosta forms for feed field.
	 *
	 * @return array
	 */
	public function forms_for_feed_setting() {

		$forms = array(
			array(
				'label' => esc_html__( 'Select a Form', 'gravityformslaposta' ),
				'value' => '',
			),
		);

		// If LaPosta API credentials are invalid, return the forms array.
		if ( ! $this->initialize_api() ) {
			return $forms;
		}

		// Get list ID.
		$list_id = $this->get_setting( 'list' );

		// Get available LaPosta forms.
		$ac_forms = $this->api->get_forms();

		// Add LaPosta forms to array and return it.
		if ( ! empty( $ac_forms ) ) {

			foreach ( $ac_forms as $form ) {

				if ( ! is_array( $form ) ) {
					continue;
				}

				if ( $form['sendoptin'] == 0 || ! is_array( $form['lists'] ) || ! in_array( $list_id, $form['lists'] ) ) {
					continue;
				}

				$forms[] = array(
					'label' => $form['name'],
					'value' => $form['id'],
				);

			}

		}

		return $forms;

	}

	// # HELPERS -------------------------------------------------------------------------------------------------------

	/**
	 * Create new LaPosta custom fields. (DISABLED FOR NOW!!)
	 *
	 * @since Unknown
	 *
	 * @param array|\Rocketgenius\Gravity_Forms\Settings\Fields\Base $field       Field object.
	 * @param array                                                  $field_value Posted field value.
	 *
	 * @return array
	 */
	public function create_new_custom_fields( $field = array(), $field_value = array() ) {

		global $_gaddon_posted_settings;

		// If no custom fields are set or if the API credentials are invalid, return settings.
		if ( empty( $field_value ) || ! $this->initialize_api() ) {
			return $field_value;
		}

		// Loop through each custom field, add new fields.
		foreach ( $field_value as $index => &$field ) {

			// If no custom key is set, move on.
			if ( rgblank( $field['custom_key'] ) ) {
				continue;
			}

			$custom_key = $field['custom_key'];

	/* 		$perstag = trim( $custom_key ); // Set shortcut name to custom key
			$perstag = str_replace( ' ', '_', $custom_key ); // Remove all spaces
			$perstag = preg_replace( '([^\w\d])', '', $custom_key ); // Strip all custom characters
			$perstag = strtoupper( $custom_key ); // Set to lowercase */

			$list_id = '';

			/* Prepare new field to add. */
			$custom_field = array(
				'list_id' => $list_id,
				'title' => $custom_key,
				'type'  => 1,
				'req'   => 0,
				'p[0]'  => 0
			);

			// Add new field.
			$new_field = $this->api->add_custom_field( $custom_field, $list_id);

			// Replace key for field with new shortcut name and reset custom key.
			if ($new_field->code  == 200 ) {

				$field['key']        = $new_field['fieldid'];
				$field['custom_key'] = '';

				// Update POST field to ensure front-end display is up-to-date.
				$_gaddon_posted_settings['custom_fields'][ $index ]['key']        = $new_field['fieldid'];
				$_gaddon_posted_settings['custom_fields'][ $index ]['custom_key'] = '';

				// Push to new custom fields array to update the UI.
				$this->_new_custom_fields[] = array(
					'label' => $custom_key,
					'value' => $new_field['fieldid'],
				);

			}else{
				// error
			}


		}

		return $field_value;

	}

	/**
	 * Checks validity of LaPosta API credentials and initializes API if valid.
	 *
	 * @return bool|null
	 */
	public function initialize_api() {

		if ( $this->api instanceof GF_LaPosta_API ) {
			return true;
		}

		/* Load the LaPosta API library. */
		require_once 'includes/class-gf-laposta-api.php';

		/* Get the global plugin settings */
		$settings = $this->get_saved_plugin_settings();
		
		/* If any of the account information fields are empty, return null. */
		if (rgempty( 'api_key', $settings )) {
			return null;
		}
		if(rgempty( 'api_url', $settings )){
			$settings['api_url'] = 'https://api.laposta.nl/';
		}


		$this->log_debug( __METHOD__ . "(): Validating API info for {$settings['api_url']} / {$settings['api_key']}." );

		$laposta = new GF_LaPosta_API( $settings['api_url'], $settings['api_key'] );

		try {

			/* Run API test. */
			$laposta->auth_test();

			/* Log that test passed. */
			$this->log_debug( __METHOD__ . '(): API credentials are valid.' );

			/* Assign LaPosta object to the class. */
			$this->api = $laposta;

			return true;

		} catch ( Exception $e ) {

			/* Log that test failed. */
			$this->log_error( __METHOD__ . '(): API credentials are invalid; ' . $e->getMessage() );

			return false;

		}

	}

	/**
	 * Gets the saved plugin settings, either from the database or the post request.
	 *
	 * This is a helper method that ensures the feedback callback receives the right value if the newest values
	 * are posted to the settings page.
	 *
	 * @since 1.9
	 *
	 * @return array
	 */
	private function get_saved_plugin_settings() {
		$prefix  = $this->is_gravityforms_supported( '2.5' ) ? '_gform_setting' : '_gaddon_setting';
		$api_url = rgpost( "{$prefix}_api_url" );
		$api_key = rgpost( "{$prefix}_api_key" );

		//$multi_activate = rgpost( "{$prefix}_multi_activate" );

		$settings = $this->get_plugin_settings();

		if ( ! $this->is_plugin_settings( $this->_slug ) || ! ( $api_url && $api_key ) ) {
			return $settings;
		}

		// clean up
 		$settings['api_url'] = esc_url( $api_url );
		$settings['api_key'] = $api_key; 

		return $settings;
	}

	/**
	 * Checks if API URL is valid.
	 *
	 * This method has been deprecated in favor of initialize_api, as that method throws an exception and logs the
	 * error message if the service is unable to connect. We need an API key to validate the API URL.
	 *
	 * @since unknown
	 * @deprecated 1.9
	 *
	 * @param string $api_url The API URL.
	 *
	 * @return null
	 */
	public function has_valid_api_url( $api_url ) {
		_deprecated_function( __METHOD__, '1.9' );
		return null;
	}
}
