<?php

use Concept7\LaravelQuestionnaire\Actions\Assessments\CalculateScores;
use Concept7\LaravelQuestionnaire\Enums\CalculationTypeEnum;
use Concept7\LaravelQuestionnaire\Models\Answer;
use Concept7\LaravelQuestionnaire\Models\Category;
use Concept7\LaravelQuestionnaire\Models\Question;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Concept7\LaravelQuestionnaire\Support\ScoreCalculatorFactory;

it('prioritizes custom strategies over default enum logic', function () {
    // 1. Define a custom strategy that returns a distinct value (999)
    $customStrategy = new class
    {
        public function calculate($answers)
        {
            return 999;
        }
    };

    // 2. Hijack the 'SUM' enum value.
    // This proves the Factory checks custom bindings BEFORE standard logic.
    ScoreCalculatorFactory::extend(
        CalculationTypeEnum::SUM->value,
        fn () => $customStrategy
    );

    // 3. Create a Questionnaire using the standard SUM type
    $questionnaire = Questionnaire::factory()->create([
        'calculation_type' => CalculationTypeEnum::SUM,
    ]);

    // 4. Setup Data
    $category = Category::factory()->create();
    $question = Question::factory()->for($questionnaire)->for($category)->create();
    $answer = Answer::factory()->for($question)->create(['score' => 10]);

    $payload = collect([
        'questionnaire' => $questionnaire,
        'submittedAnswers' => collect([
            ['question_id' => $question->getKey(), 'answer_id' => $answer->getKey()],
        ]),
    ]);

    // 5. Run Logic
    $result = (new CalculateScores)->handle($payload, fn ($p) => $p);

    // 6. Assert
    // If it used the default SUM, result would be 10.
    // Since we overrode it, result is 999.
    expect($result->get('scoresPerCategory')->get($category->getKey()))
        ->toBe(999);
});
