<?php

namespace Concept7\LaravelQuestionnaire\Support;

use Concept7\LaravelQuestionnaire\Contracts\ScoreCalculationStrategy;
use Concept7\LaravelQuestionnaire\Enums\CalculationTypeEnum;
use Concept7\LaravelQuestionnaire\Strategies\AverageStrategy;
use Concept7\LaravelQuestionnaire\Strategies\HighestScoreStrategy;
use Concept7\LaravelQuestionnaire\Strategies\SumStrategy;
use Concept7\LaravelQuestionnaire\Strategies\WeightedAverageStrategy;

class ScoreCalculatorFactory
{
    /** @var array<string, \Closure> */
    protected static $customStrategies = [];

    /**
     * Flush the custom strategies.
     */
    public static function flush(): void
    {
        static::$customStrategies = [];
    }

    public static function extend(string $type, \Closure $callback): void
    {
        static::$customStrategies[$type] = $callback;
    }

    /** @var array<string, class-string> */
    protected static array $strategies = [
        'average' => AverageStrategy::class,
        'highest_score' => HighestScoreStrategy::class,
        'weighted_average' => WeightedAverageStrategy::class,
        'sum' => SumStrategy::class,
    ];

    /**
     * @param  class-string  $strategyClass
     */
    public static function register(CalculationTypeEnum $type, string $strategyClass): void
    {
        self::$strategies[$type->value] = $strategyClass;
    }

    /**
     * @return ScoreCalculationStrategy
     */
    public static function make(?CalculationTypeEnum $type): mixed
    {
        $key = $type?->value ?? 'sum';

        if (isset(static::$customStrategies[$key])) {
            return call_user_func(static::$customStrategies[$key]);
        }

        $class = self::$strategies[$key] ?? self::$strategies['sum'];

        return app($class);
    }
}
