<?php

use Concept7\LaravelQuestionnaire\Actions\Assessments\DetermineAndProcessResults;
use Concept7\LaravelQuestionnaire\Models\Assessment;
use Concept7\LaravelQuestionnaire\Models\Category;
use Concept7\LaravelQuestionnaire\Models\Interpretation;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Concept7\LaravelQuestionnaire\Models\Scale;
use Concept7\LaravelQuestionnaire\Models\User;
use Illuminate\Support\Collection;

beforeEach(function(){

    config()->set('questionnaire.models.user', User::class);

    $this->questionnaire = Questionnaire::factory()->create();
    $this->assessment = Assessment::factory()
        ->for(User::factory())
        ->for($this->questionnaire)
        ->create();
    $this->category1 = Category::factory()->create();
    $this->category2 = Category::factory()->create();
    $this->scaleLow = Scale::factory()->create(['name' => 'Low']);
    $this->scaleHigh = Scale::factory()->create(['name' => 'High']);

    // Rule 1: Category 1, Score 0-20 is "Low"
    Interpretation::factory()->create([
        'category_id' => $this->category1->id,
        'scale_id' => $this->scaleLow->id,
        'min_score' => 0,
        'max_score' => 20,
    ]);

    // Rule 2: Category 1, Score 21-40 is "High"
    Interpretation::factory()->create([
        'category_id' => $this->category1->getKey(),
        'scale_id' => $this->scaleHigh->getKey(),
        'min_score' => 21,
        'max_score' => 40,
    ]);

    // Rule 3: Category 2, Score 0-50 is "Low"
    Interpretation::factory()->create([
        'category_id' => $this->category2->getKey(),
        'scale_id' => $this->scaleLow->getKey(),
        'min_score' => 0,
        'max_score' => 50,
    ]);

});



it('correctly determines and saves results for all categories', function(){
    $payload = collect(
        [
            'assessment' => $this->assessment,
            'scoresPerCategory' => collect([
                $this->category1->getKey() => 30, //high
                $this->category2->getKey() => 10, // low
            ])
        ]
    );

    $pipe = new DetermineAndProcessResults();
    $next = fn($payload) => $payload;

    $resultPayload = $pipe->handle($payload, $next);

    $this->assertDatabaseHas('assessment_results', [
        'assessment_id' => $this->assessment->getKey(),
        'category_id' => $this->category1->getKey(),
        'scale_id' => $this->scaleHigh->getKey(),
        'total_score' => 30, //high
    ]);

    $this->assertDatabaseHas('assessment_results', [
        'assessment_id' => $this->assessment->getKey(),
        'category_id' => $this->category2->getKey(),
        'scale_id' => $this->scaleLow->getKey(),
        'total_score' => 10, //low
    ]);

    expect($resultPayload->get('assessmentResults'))
        ->toBeInstanceOf(Collection::class)
        ->toHaveCount(2);
});

it('correctly handles boundary scores', function(){
   $payload = collect([
       'assessment' => $this->assessment,
       'scoresPerCategory' => collect([
           $this->category1->getKey() => 20, // Exactly on the max_score for "Low"
           $this->category2->getKey() => 50, // Exactly on the max_score for "Low"
       ])
   ]) ;

    $pipe = new DetermineAndProcessResults();
    $next = fn($payload) => $payload;

    $pipe->handle($payload, $next);

    $this->assertDatabaseHas('assessment_results', [
        'assessment_id' => $this->assessment->getKey(),
        'category_id' => $this->category1->getKey(),
        'scale_id' => $this->scaleLow->getKey(),
        'total_score' => 20,
    ]);

    // Assert "Low" for Cat 2
    $this->assertDatabaseHas('assessment_results', [
        'assessment_id' => $this->assessment->getKey(),
        'category_id' => $this->category2->getKey(),
        'scale_id' => $this->scaleLow->getKey(),
        'total_score' => 50,
    ]);

});


it('correctly handles boundary scores on the next rule', function () {
    $payload = collect([
        'assessment' => $this->assessment,
        'scoresPerCategory' => collect([
            $this->category1->getKey() => 21, // Exactly on the min_score for "High"
            $this->category2->getKey() => 0,  // Exactly on the min_score for "Low"
        ])
    ]);

    $pipe = new DetermineAndProcessResults();
    $next = fn($payload) => $payload;

    $pipe->handle($payload, $next);

    // Assert "High" for Cat 1
    $this->assertDatabaseHas('assessment_results', [
        'assessment_id' => $this->assessment->getKey(),
        'category_id' => $this->category1->getKey(),
        'scale_id' => $this->scaleHigh->getKey(),
        'total_score' => 21,
    ]);

    // Assert "Low" for Cat 2
    $this->assertDatabaseHas('assessment_results', [
        'assessment_id' => $this->assessment->getKey(),
        'category_id' => $this->category2->getKey(),
        'scale_id' => $this->scaleLow->getKey(),
        'total_score' => 0,
    ]);
});
