<?php

use Concept7\LaravelQuestionnaire\Models\Category;
use Concept7\LaravelQuestionnaire\Models\Question;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use function Pest\Laravel\assertDatabaseHas;
use function Pest\Laravel\assertDatabaseMissing;

beforeEach(function(){
    $categories = Category::factory()
        ->count(5)
        ->create()
        ->pluck('id');

    $this->questionnaire = Questionnaire::factory()->create();

    Question::factory()
        ->count(5)
        ->for($this->questionnaire)
        ->state(function (array $attributes) use ($categories) {
            return [
                'category_id' => $categories->random(),
            ];
        })
        ->hasAnswers(5)
        ->create();
});

describe('index questions', function(){
    test('it returns a list of questions', function(){
        $questions = Question::all();
        $this->assertCount(5, $questions);
    });

    test('questions are ordered by order column', function(){
        $questions = Question::all();

        $firstQuestion = $questions->first();
        $lastQuestion = $questions->last();

        expect($firstQuestion->order)->toBeLessThan($lastQuestion->order);
    });
});

describe('CRUD questions', function(){
   it('can create a question', function(){
       $attributes = ['text' => 'Test Question',];

       $question = Question::factory()
           ->for(Questionnaire::factory())
           ->for(Category::factory())
           ->state($attributes)
           ->create();

       expect($question)->toBeInstanceOf(Question::class);
       assertDatabaseHas('questions', $attributes);
   }) ;

   it('can update a question', function(){
       $question = Question::factory()
           ->for(Questionnaire::factory())
           ->for(Category::factory())
           ->create();
       $question->update(['text' => 'Updated Question']);

       assertDatabaseHas('questions', [
           'text' => 'Updated Question',
       ]);
   });

   it('can delete a question', function(){
       $question = Question::factory()
           ->for(Questionnaire::factory())
           ->for(Category::factory())
           ->create();

       $question->delete();

       assertDatabaseMissing('questions', [
           'id' => $question->id,
       ]);
   });

   it('sorts questions when a new question is created', function(){
       $category = Category::factory()->create();

       Question::factory()
           ->for($this->questionnaire)
           ->for($category)
           ->count(3)
           ->create();

       $lastQuestionOrder = $category->questions->last()->order;

       $newQuestion = Question::factory()
           ->for($this->questionnaire)
           ->for($category)
           ->create();

       expect($lastQuestionOrder)->toBeLessThan($newQuestion->order);
   });
});
