<?php

use Concept7\LaravelQuestionnaire\Models\Answer;
use Concept7\LaravelQuestionnaire\Models\Category;
use Concept7\LaravelQuestionnaire\Models\Question;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use function Pest\Laravel\assertDatabaseHas;
use function Pest\Laravel\assertDatabaseMissing;

beforeEach(function(){
    $categories = Category::factory()
        ->count(5)
        ->create()
        ->pluck('id');

    $questionnaire = Questionnaire::factory()->create();

    Question::factory()
        ->count(5)
        ->state(function (array $attributes) use ($categories,$questionnaire) {
            return [
                'questionnaire_id' => $questionnaire->getKey(),
                'category_id' => $categories->random(),
            ];
        })
        ->hasAnswers(5)
        ->create();
});

describe('index answers', function(){
    test('it returns a list of answers', function(){
        $answers = Answer::all();
        $this->assertCount(25, $answers);
    });

    test('answers are ordered by order column', function(){
        $answers = Answer::all();

        $firstAnswer = $answers->first();
        $lastAnswer = $answers->last();

        expect($firstAnswer->order)->toBeLessThan($lastAnswer->order);
    });
});

describe('we can crud for answers', function(){
    it('can create a answer', function(){
        $attributes = ['text' => 'Test answer'];

        $question = Question::factory()
            ->for(Questionnaire::factory())
            ->for(Category::factory())
            ->state($attributes)
            ->create();

        $attributes['question_id'] = $question->getKey();

        $answer = Answer::factory()->state($attributes)->create();

        expect($answer)->toBeInstanceOf(Answer::class);
        assertDatabaseHas('answers', $attributes);
    }) ;

    it('can update a answer', function(){
        $answer = Answer::factory()
            ->for(Question::factory()
                ->for(Questionnaire::factory())
                ->for(Category::factory())
            )
            ->create();
        $answer->update(['text' => 'Updated answer']);
        assertDatabaseHas('answers', [
            'text' => 'Updated answer',
        ]);
    });

    it('can delete a answer', function(){
        $attributes = ['text' => 'Test answer',];
        $answer = Answer::factory()
            ->state($attributes)
            ->for(
                Question::factory()
                    ->for(Questionnaire::factory())
                    ->for(Category::factory())
            )
            ->create();

        $answer->delete();

        assertDatabaseMissing( 'answers', $attributes);
    });

    it('sorts answers when a new answer is created', function(){
        $question = Question::factory()
            ->for(Questionnaire::factory())
            ->for(Category::factory())
            ->hasAnswers(4)
            ->create();
        $lastAnswerOrder = $question->answers->last()->order;
        $newAnswer = Answer::factory()->for($question)->create();

        expect($lastAnswerOrder)->toBeLessThan($newAnswer->order);
    });
});
