<?php
namespace Concept7\LaravelQuestionnaire;

use Concept7\LaravelQuestionnaire\Http\Middleware\EnsureAssessmentAccess;
use Spatie\LaravelPackageTools\Commands\InstallCommand;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;
use Concept7\LaravelQuestionnaire\Console\Commands\GenerateTestDataCommand;
use Illuminate\Foundation\Http\Middleware\HandlePrecognitiveRequests;

class LaravelQuestionnaireServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        /*
         * This name will be used to load config and views
         * and for publishing assets.
         */
        $package
            ->name("laravel-questionnaire")
            ->hasConfigFile()
            ->hasViews("laravel-questionnaire")
            ->hasViewComponents("laravel-questionnaire")
            ->hasTranslations()
            ->discoversMigrations(path: "database/migrations")
            ->hasRoute("web")
            ->hasAssets()
            ->hasCommands([GenerateTestDataCommand::class])
            ->hasInstallCommand(function (InstallCommand $command) {
                $command
                    ->publishConfigFile()
                    ->publishMigrations()
                    ->askToRunMigrations()
                    ->publishAssets()
                    ->copyAndRegisterServiceProviderInApp();
            });
    }

    public function bootingPackage(): void
    {
        $router = $this->app->make("router");

        //conditionally enable laravel precognition: https://laravel.com/docs/12.x/precognition#using-alpine
        if (str(config("questionnaire.mode"))->contains("alpine")) {
            $router->pushMiddlewareToGroup(
                "web",
                HandlePrecognitiveRequests::class,
            );
        }

        $router->aliasMiddleware(
            "questionnaire.access",
            EnsureAssessmentAccess::class,
        );

        $router->bind("questionnaire", function ($value) {
            $modelClass = config("questionnaire.models.questionnaire");

            if (!class_exists($modelClass)) {
                throw new \Exception(
                    "Questionnaire model class $modelClass not found",
                );
            }

            return $modelClass::where("slug", $value)->firstOrFail();
        });

        $router->bind("assessment", function ($value) {
            $modelClass = config("questionnaire.models.assessment");

            if (!class_exists($modelClass)) {
                throw new \Exception(
                    "Assessment model class $modelClass not found",
                );
            }

            return $modelClass::where("id", $value)->firstOrFail();
        });

        $router->bind("user", function ($value) {
            $modelClass = config("questionnaire.models.user");

            if (!class_exists($modelClass)) {
                throw new \Exception("User model class $modelClass not found");
            }

            return $modelClass::where("id", $value)->firstOrFail();
        });
    }
}
