<?php

namespace Concept7\LaravelQuestionnaire\Helpers;

use Concept7\LaravelQuestionnaire\Models\Assessment;
use Concept7\LaravelQuestionnaire\Models\AssessmentResult;
use Concept7\LaravelQuestionnaire\Models\Category;
use Illuminate\Support\Collection;

class ChartHelper
{
    /**
     * Build the Chart.js data structure.
     *
     * Expects $labels and $data to be Collections to leverage map/values operations.
     */
    public static function getChartData(Collection $labels, Collection $data): array
    {
        return  [
            'labels' => $labels,
            'datasets' => [[
                'label' => __('Total score per category'),
                'data' => $data,
                'backgroundColor' => $labels->map(fn () => 'rgba(54, 162, 235, 0.5)'),
                'borderColor' => $labels->map(fn () => 'rgba(54, 162, 235, 1)'),
                'borderWidth' => 1,
            ]],
        ];
    }

    /**
     * Default Chart.js options including axis titles.
     */
    public static function getDefaultChartOptions(?string $xTitle = null, ?string $yTitle = null): array
    {
        $xTitle = $xTitle ?? __('Category — Scale');
        $yTitle = $yTitle ?? __('Score');

        return [
            'responsive' => true,
            'maintainAspectRatio' => false,
            'scales' => [
                'x' => [
                    'title' => [
                        'display' => true,
                        'text' => $xTitle,
                    ],
                ],
                'y' => [
                    'beginAtZero' => true,
                    'title' => [
                        'display' => true,
                        'text' => $yTitle,
                    ],
                ],
            ],
            'plugins' => [
                'legend' => [
                    'display' => true,
                    'position' => 'top',
                ],
                'tooltip' => [
                    'enabled' => true,
                ],
            ],
        ];
    }

    /**
     * Build Chart.js data by querying assessment_results for a given assessment.
     *
     * Returns null if no results are found.
     */
    public static function buildChartDataFromAssessment(Assessment $assessment): ?array
    {
        $resultClass = config('questionnaire.models.assessment_result', AssessmentResult::class);

        $results = $resultClass::withoutGlobalScopes()
            ->with([
                'category' => fn($q) => $q->withoutGlobalScopes()->select('id', 'name'),
                'scale'    => fn($q) => $q->withoutGlobalScopes()->select('id', 'name')
            ])
            ->where('assessment_id', $assessment->getKey())
            ->orderBy('category_id')
            ->get(['category_id', 'total_score', 'scale_id']);

        if ($results->isEmpty()) {
            return null;
        }

        $labels = $results->map(function ($r) {
            $categoryName = optional($r->category)->name ?? "Category #{$r->category_id}";
            $scaleName = optional($r->scale)->name ?? __('No scale');
            return sprintf('%s — %s', $categoryName, $scaleName);
        });


        $data = $results->pluck('total_score');
        return self::getChartData($labels->values(), $data->values());
    }

    /**
     * Build Chart.js data from a scores-per-category collection.
     *
     * Kept for backward compatibility; prefer using buildChartDataFromAssessment().
     * The $scores collection should be of the form: [categoryId => totalScore].
     * Returns null if the collection is empty.
     */
    public static function buildChartData(?Collection $scores): ?array
    {
        if (!$scores || $scores->isEmpty()) {
            return null;
        }

        $categoryIds = $scores->keys()->all();
        $categories = Category::whereIn('id', $categoryIds)->pluck('name', 'id');

        $labels = $scores->keys()->map(fn ($catId) => $categories[$catId] ?? "Category #{$catId}")->values();
        $data = $scores->values();

        return self::getChartData($labels, $data);
    }
}
