<?php

namespace Concept7\LaravelQuestionnaire\Actions\Assessments;

use Closure;
use Concept7\LaravelQuestionnaire\Enums\CalculationTypeEnum;
use Concept7\LaravelQuestionnaire\Models\Answer;
use Illuminate\Support\Collection;

class CalculateScores
{
    public function handle(Collection $payload, Closure $next)
    {
        $questionnaire = $payload->get('questionnaire');
        $submittedAnswers = $payload->get('submittedAnswers');
        $answerIds = $submittedAnswers->pluck('answer_id');

        $answers = Answer::with('question.answers')
            ->whereIn('id', $answerIds)
            ->get();

        $scoresPerCategory = $answers
            ->groupBy('question.category_id')
            ->map(function (Collection $answersInCategory) use ($questionnaire) {
                return match ($questionnaire->calculation_type) {
                    CalculationTypeEnum::AVERAGE => $answersInCategory->avg('score'),
                    CalculationTypeEnum::HIGHEST_SCORE => $answersInCategory->max('score'),
                    CalculationTypeEnum::WEIGHTED_AVERAGE => $this->calculateWeightedScore($answersInCategory),

                    default => $answersInCategory->sum('score'),
                };
        });

        $payload->put('scoresPerCategory', $scoresPerCategory);

        return $next($payload);
    }

    public function calculateWeightedScore(Collection $answersInCategory): float
    {
        $totalAchievedPoints = 0;
        $totalPossiblePoints = 0;

        foreach ($answersInCategory as $answer) {
            $question = $answer->question;

            if(!$question) {
                dd('Question relation is NULL for answer ID: ' . $answer->id);
            }

            $weight = $question->weight ?? 1;
            $totalAchievedPoints += $answer->score * $weight;

            $maxScore = $question->answers->max('score');

            if($maxScore === null) {
                $maxScore = 10;
            }

            $totalPossiblePoints += $maxScore * $weight;
        }

        if ($totalPossiblePoints === 0) {
            return 0;
        }

        return round(($totalAchievedPoints / $totalPossiblePoints) * 100, 2);
    }
}
