<?php

use Concept7\LaravelQuestionnaire\Enums\CalculationTypeEnum;
use Concept7\LaravelQuestionnaire\Enums\InputTypeEnum;
use Concept7\LaravelQuestionnaire\Models\Answer;
use Concept7\LaravelQuestionnaire\Models\Assessment;
use Concept7\LaravelQuestionnaire\Models\Category;
use Concept7\LaravelQuestionnaire\Models\Interpretation;
use Concept7\LaravelQuestionnaire\Models\Question;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Concept7\LaravelQuestionnaire\Models\Scale;
use Concept7\LaravelQuestionnaire\Models\User;
use Illuminate\Support\Facades\URL;

beforeEach(function () {
    config()->set('questionnaire.models.participant', User::class);
    config()->set('questionnaire.models.assessment', Assessment::class);
    config()->set('questionnaire.models.questionnaire', Questionnaire::class);

    config()->set('questionnaire.thank_you_page_enabled', false);

    // config()->set('questionnaire.middleware.assessments.create', ['web']);
    // config()->set('questionnaire.middleware.assessments.show', ['web']);
});

test('it can go to the assessment route (anonymous)', function () {
    // $this->withoutExceptionHandling(); // <--- Add this line

    config()->set('questionnaire.mode', 'default');
    config()->set('questionnaire.allow_public_access', true);

    $categories = Category::factory()->count(5)->create();

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($categories->random())
                ->has(Answer::factory()
                    ->state(fn () => ['score' => 50])
                    ->count(5))
        )
        ->create();

    // Generate a signed URL for anonymous access (participant is null/missing)
    $url = route('questionnaire.assessments.create', [
        'questionnaire' => $questionnaire->slug,
    ]);

    $response = $this->get($url);

    $response->assertStatus(200);
});

it('expects questions posted (anonymous)', function () {
    config()->set('questionnaire.mode', 'default');
    config()->set('questionnaire.allow_public_access', true);

    $categories = Category::factory()->count(5)->create();

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($categories->random())
                ->has(Answer::factory()
                    ->state(fn () => ['score' => 50])
                    ->count(5))
        )
        ->create();

    $url = URL::signedRoute('questionnaire.assessments.store', [
        'questionnaire' => $questionnaire->slug,
    ]);

    $response = $this->post($url);

    // Should fail validation (302) not Auth (403)
    $response->assertStatus(302);
    $response->assertSessionHasErrors();
});

it('expects a successfully response when posting questions', function () {
    config()->set('questionnaire.mode', 'default');
    config()->set('questionnaire.models.participant', \Concept7\LaravelQuestionnaire\Models\User::class);
    config()->set('questionnaire.participant_url_key', 'id');

    $categories = Category::factory()->count(5)->create();
    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($categories->random())
                ->has(Answer::factory()
                    ->state(fn () => ['score' => 50])
                    ->count(5))
        )
        ->create();

    $usedCategories = $questionnaire->questions->pluck('category_id')->unique();

    foreach ($usedCategories as $categoryId) {
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleLow->getKey(),
            'min_score' => 0,
            'max_score' => 20,
        ]);
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleHigh->getKey(),
            'min_score' => 21,
            'max_score' => 100,
        ]);
    }

    $postData = $questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                ? [$question->answers->random()->getKey()]
                : $question->answers->random()->getKey(),
        ];
    })->all();

    $participant = User::factory()->create();

    $response = $this->post(route('questionnaire.assessments.store', [
        'questionnaire' => $questionnaire->slug,
        'participant' => $participant->getKey(),
    ]), $postData);

    $response->assertStatus(302);
});

it('fails validation when a wrong answer is posted that does not belong to the question', function () {
    config()->set('questionnaire.mode', 'default');

    $participant = User::factory()->create();
    $categories = Category::factory()->count(5)->create();

    $questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($categories->random())
                ->has(Answer::factory()
                    ->state(fn () => ['score' => 50])
                    ->count(5))
        )
        ->create();

    $wrongQuestion = Question::factory()
        ->for($categories->random())
        ->for($questionnaire)
        ->has(Answer::factory()->count(5))
        ->create();

    $usedCategories = $questionnaire->questions->pluck('category_id')->unique();

    $postData = $questionnaire->questions->mapWithKeys(function ($question) use ($wrongQuestion) {
        return [
            'questions_'.$wrongQuestion->getKey() => $wrongQuestion->type === InputTypeEnum::CHECKBOX
                ? [$wrongQuestion->answers->random()->getKey()]
                : $wrongQuestion->answers->random()->getKey(),
        ];
    })->all();

    $postData['_token'] = 'fake-csrf-token-for-testing';

    $url = route('questionnaire.assessments.store', [
        'questionnaire' => $questionnaire->slug,
        'participant' => $participant->getKey(),
    ]);

    $response = $this->post($url, $postData);

    $response->assertStatus(302);
});

it('expects assessment in the database and stores answers (based on anonymous participant)', function () {
    config()->set('questionnaire.mode', 'default');
    config()->set('questionnaire.allow_public_access', true);
    config()->set('questionnaire.thank_you_page_enabled', true);

    $this->assertDatabaseCount('assessments', 0);

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);
    $categories = Category::factory()->count(5)->create();

    $questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($categories->random())
                ->has(Answer::factory()
                    ->state(fn () => ['score' => 50])
                    ->count(5))
        )
        ->create();

    $usedCategories = $questionnaire->questions->pluck('category_id')->unique();

    foreach ($usedCategories as $categoryId) {
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleLow->getKey(),
            'min_score' => 0,
            'max_score' => 20,
        ]);
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleHigh->id,
            'min_score' => 21,
            'max_score' => 100,
        ]);
    }

    $postDataCollection = $questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                ? [$question->answers->random()->getKey()]
                : $question->answers->random()->getKey(),
        ];
    });

    $firstQuestionKey = 'questions_'.$questionnaire->questions->first()->getKey();
    $expectedAnswerId = $postDataCollection[$firstQuestionKey];

    $response = $this->post(
        route('questionnaire.assessments.store', [
            'questionnaire' => $questionnaire->slug,
        ]),
        $postDataCollection->all(),
    );

    $response->assertStatus(302);

    $this->assertDatabaseCount('assessments', 1);

    $assessment = Assessment::latest()->first();

    expect($assessment)->not->toBeNull();

    $this->assertEquals($questionnaire->getKey(), $assessment->questionnaire_id);

    $this->assertDatabaseHas('assessment_answers', [
        'answer_id' => $expectedAnswerId,
        'assessment_id' => $assessment->getKey(),
    ]);
});

it('expects assessment in the database and stores answers (based on linked participant)', function () {
    config()->set('questionnaire.mode', 'default');
    config()->set('questionnaire.allow_public_access', false);

    $this->assertDatabaseCount('assessments', 0);

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $categories = Category::factory()->count(5)->create();
    $questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($categories->random())
                ->has(Answer::factory()
                    ->state(fn () => ['score' => 50])
                    ->count(5))
        )
        ->create();

    $participant = User::factory()->create();

    foreach ($categories as $categoryId) {
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleLow->getKey(),
            'min_score' => 0,
            'max_score' => 20,
        ]);
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleHigh->id,
            'min_score' => 21,
            'max_score' => 100,
        ]);
    }

    $postDataCollection = $questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                ? [$question->answers->random()->getKey()]
                : $question->answers->random()->getKey(),
        ];
    });

    $url = route('questionnaire.assessments.store', [
        'questionnaire' => $questionnaire->slug,
        'participant' => $participant->getKey(),
    ]);

    $firstQuestionKey = 'questions_'.$questionnaire->questions->first()->getKey();
    $expectedAnswerId = $postDataCollection[$firstQuestionKey];

    $response = $this->post($url, $postDataCollection->all());

    $response->assertStatus(302);

    $this->assertDatabaseCount('assessments', 1);

    $assessment = Assessment::latest()->first();

    expect($assessment)->not->toBeNull();

    $this->assertEquals($participant->getKey(), $assessment->assessmentable_id);

    $this->assertEquals($participant->getKey(), $assessment->questionnaire_id);

    $this->assertDatabaseHas('assessment_answers', [
        'answer_id' => $expectedAnswerId,
        'assessment_id' => $assessment->getKey(),
    ]);

    $this->assertDatabaseHas('assessments', [
        'assessmentable_id' => $participant->getKey(),
        'assessmentable_type' => User::class,
    ]);
});

describe('assessment show template', function () {
    it('loads corresponding template', function ($calculationType, $assertViewIs) {
        config()->set('questionnaire.layout', 'laravel-questionnaire::layouts.app');

        $categories = Category::factory()->count(5)->create();

        $participant = User::factory()->create();

        $questionnaire = Questionnaire::factory(['calculation_type' => $calculationType])
            ->has(
                Question::factory()
                    ->for($categories->random())
                    ->has(Answer::factory()
                        ->state(fn () => ['score' => 50])
                        ->count(5))
            )
            ->create();

        $scaleLow = Scale::factory()->create(['name' => 'Low']);
        $scaleHigh = Scale::factory()->create(['name' => 'High']);

        $usedCategories = $questionnaire->questions->pluck('category_id')->unique();

        foreach ($usedCategories as $categoryId) {
            Interpretation::factory()->create([
                'category_id' => $categoryId,
                'scale_id' => $scaleLow->getKey(),
                'min_score' => 0,
                'max_score' => 20,
            ]);
            Interpretation::factory()->create([
                'category_id' => $categoryId,
                'scale_id' => $scaleHigh->id,
                'min_score' => 21,
                'max_score' => 100,
            ]);
        }

        $postDataCollection = $questionnaire->questions->mapWithKeys(function ($question) {
            return [
                'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                    ? [$question->answers->random()->getKey()]
                    : $question->answers->random()->getKey(),
            ];
        });

        $response = $this->post(
            route('questionnaire.assessments.store', [
                'questionnaire' => $questionnaire->slug,
                'participant' => $participant->getKey(),
            ]),
            $postDataCollection->all()
        );

        $response->assertStatus(302);

        $assessment = Assessment::where('questionnaire_id', $questionnaire->getKey())
            ->latest()
            ->first();

        $response->assertRedirect(route('questionnaire.assessments.show', [
            'questionnaire' => $questionnaire->slug,
            'assessment' => $assessment->getKey(),
            'participant' => $participant->getKey(),
        ]));

        $response = $this->get(route('questionnaire.assessments.show', [
            'questionnaire' => $questionnaire->slug,
            'assessment' => $assessment->getKey(),
            'participant' => $participant->getKey(),
        ]));

        $response->assertStatus(200);

        $response->assertViewIs($assertViewIs);
    })->with([
        'it loads default template' => [
            'calculationType' => CalculationTypeEnum::AVERAGE,
            'assertViewIs' => 'laravel-questionnaire::assessments.results.average',
        ],
        'it loads sum template' => [
            'calculationType' => CalculationTypeEnum::SUM,
            'assertViewIs' => 'laravel-questionnaire::assessments.results.sum',
        ],
    ]);
});

describe('assessment create template', function () {
    it('loads corresponding template', function ($mode, $assertViewIs) {
        config()->set('questionnaire.mode', $mode);

        $categories = Category::factory()->count(5)->create();

        $questionnaire = Questionnaire::factory()
            ->has(
                Question::factory()
                    ->for($categories->random())
                    ->has(Answer::factory()
                        ->state(fn () => ['score' => 50])
                        ->count(5))
            )
            ->create();

        $response = $this->get(route('questionnaire.assessments.create', ['questionnaire' => $questionnaire->slug]));

        $response->assertViewIs($assertViewIs);
    })->with([
        'it loads default template' => ['mode' => 'default', 'assertViewIs' => 'laravel-questionnaire::assessments.create'],
        'it loads alpine wizard template' => ['mode' => 'alpine', 'assertViewIs' => 'laravel-questionnaire::assessments.wizard.create'],
        'it loads alpine_api wizard template' => ['mode' => 'alpine_via_api', 'assertViewIs' => 'laravel-questionnaire::assessments.wizard.create_api'],
    ]);
});

it('redirects to thank_you page instead of show route', function () {
    config()->set('questionnaire.thank_you_page_enabled', true);

    $category = Category::factory()->create();

    $questionnaire = Questionnaire::factory(['calculation_type' => CalculationTypeEnum::AVERAGE])
        ->has(
            Question::factory()
                ->for($category)
                ->has(Answer::factory()
                    ->count(5))
        )
        ->create();

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $usedCategories = $questionnaire->questions->pluck('category_id')->unique();

    foreach ($usedCategories as $categoryId) {
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleLow->getKey(),
            'min_score' => 0,
            'max_score' => 20,
        ]);
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleHigh->id,
            'min_score' => 21,
            'max_score' => 100,
        ]);
    }

    $postDataCollection = $questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                ? [$question->answers->random()->getKey()]
                : $question->answers->random()->getKey(),
        ];
    });

    $response = $this->post(
        route('questionnaire.assessments.store', ['questionnaire' => $questionnaire->slug]),
        $postDataCollection->all(),
    );

    $response->assertRedirect(route('questionnaire.thankyou', $questionnaire->slug));
});
