@extends(config('questionnaire.layout', 'layouts.app'))

@section('content')
    <div x-data="app()" x-cloak>
        <form>
            @csrf
            <div class="max-w-3xl mx-auto px-4 py-10">

                <div x-show.transition="step === 'complete'">
                    DONE
                </div>

                <div x-show.transition="step != 'complete'">
                    <div class="uppercase tracking-wide text-xs font-bold text-gray-500 mb-1 leading-tight"
                         x-text="`Step: ${step} of ${totalQuestions}`"
                    ></div>

                    @if( config('questionnaire.with_progress_bar') )
                        <div
                            class="progress-bar bg-blue-500 h-2.5 rounded-full mb-4"
                            :style="{ width: progressPercentage + '%'  }"
                        ></div>
                    @endif

                    <div class="flex flex-col md:flex-row md:items-center md:justify-between">
                        <div class="flex-1">

                            <template x-if="currentQuestion">
                                <div
                                    :key="currentQuestion.id"
                                    x-transition:enter="transition ease-out duration-300"
                                    x-transition:enter-start="opacity-0 transform -translate-x-4"
                                    x-transition:enter-end="opacity-100 transform translate-x-0"
                                    x-transition:leave="transition ease-in duration-200"
                                    x-transition:leave-start="opacity-100 transform translate-x-0"
                                    x-transition:leave-end="opacity-0 transform translate-x-4"
                                >
                                    <h3 class="text-2xl font-semibold text-gray-900 mb-6"
                                        x-text="currentQuestion.text"></h3>
                                    <div class="gap-y-3">
                                        <template x-for="answer in currentQuestion.answers" :key="answer.id">
                                            <button
                                                @click="selectAnswer(answer.id)"
                                                type="button"
                                                class="w-full cursor-pointer text-left p-4 bg-white border border-gray-300 rounded-lg shadow-sm hover:bg-gray-50 hover:border-blue-500 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50 transition duration-150 ease-in-out">
                                                <span class="text-lg text-gray-700" x-text="answer.text"></span>
                                            </button>
                                        </template>
                                    </div>
                                </div>

                            </template>

                            <!-- Loading/Error State -->
                            <template x-if="!currentQuestion && step !== 'complete'">
                                <div class="text-gray-500">Loading question...</div>
                            </template>

                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
@endsection
@push('scripts')
    <script>
        function app() {
            return {
                step: 1,
                questions: @json($questions),
                givenAnswers: {},

                get totalQuestions() {
                    return this.questions ? this.questions.length : 0;
                },

                get progressPercentage(){
                    if(this.totalQuestions === 0){
                        return 0;
                    }
                    return (this.step / this.totalQuestions) * 100;
                },

                get currentQuestion() {
                    if (this.step > 0 && this.step <= this.totalQuestions) {
                        return this.questions[this.step - 1];
                    }
                    return null;
                },

                selectAnswer(answerId) {
                    if (!this.currentQuestion) return;

                    const questionId = this.currentQuestion.id;

                    this.givenAnswers[`questions_${questionId}`] = answerId;

                    if (this.step < this.totalQuestions) {
                        this.step++;
                    } else {
                        this.step = 'complete';
                        this.submitForm();
                    }
                },

                async submitForm() {
                    const csrfToken = document.querySelector('[name=_token]').value;
                    const storeUrl = '{{ route("questionnaire.assessments.store", ['questionnaire' => $questionnaire->slug])  }}';

                    try {
                        const response = await fetch(storeUrl, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': csrfToken,
                                'Accept': 'application/json'
                            },
                            body: JSON.stringify(this.givenAnswers)
                        })

                        if (response.ok) {
                            const result = await response.json();
                            if (result.redirect_url) {
                                window.location.href = result.redirect_url;
                            }
                        } else {
                            const errorData = await response.json();
                            console.error('Submission failed', response.statusText, errorData);
                        }
                    } catch (error) {
                        console.error('Error submitting the form', error);
                    }
                }
            }
        }
    </script>
@endpush
