<?php

use Concept7\LaravelQuestionnaire\Enums\CalculationTypeEnum;
use Concept7\LaravelQuestionnaire\Enums\InputTypeEnum;
use Concept7\LaravelQuestionnaire\Models\Answer;
use Concept7\LaravelQuestionnaire\Models\Assessment;
use Concept7\LaravelQuestionnaire\Models\Category;
use Concept7\LaravelQuestionnaire\Models\Interpretation;
use Concept7\LaravelQuestionnaire\Models\Question;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Concept7\LaravelQuestionnaire\Models\Scale;
use Concept7\LaravelQuestionnaire\Models\User;
use Concept7\LaravelQuestionnaire\Support\QuestionnaireContext;

beforeEach(function () {
    config()->set('questionnaire.models.participant', User::class);
    config()->set('questionnaire.models.assessment', Assessment::class);
    config()->set('questionnaire.models.questionnaire', Questionnaire::class);

    config()->set('questionnaire.thank_you_page_enabled', false);

    config()->set('questionnaire.middleware.assessments.create', ['web']);
    config()->set('questionnaire.middleware.assessments.show', ['web']);

    $this->participant = User::factory()->create();
    $this->categories = Category::factory()->count(5)->create();

    $this->questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($this->categories->random())
                ->has(Answer::factory()->count(5))
                ->count(5)
        )
        ->create();
});

test('it can go to the assessment route', function () {
    config()->set('questionnaire.mode', 'default');

    $response = $this->get(route('questionnaire.assessments.create', [
        'questionnaire' => $this->questionnaire->slug,
    ]));

    $response->assertStatus(200);
});

it('expects questions posted', function () {
    config()->set('questionnaire.mode', 'default');

    $response = $this->post(route('questionnaire.assessments.store', ['questionnaire' => $this->questionnaire->slug]));
    $response->assertStatus(302);
});

it('expects a successfully response when posting questions', function () {
    config()->set('questionnaire.mode', 'default');

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $usedCategories = $this->questionnaire->questions->pluck('category_id')->unique();

    foreach ($usedCategories as $categoryId) {
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleLow->getKey(),
            'min_score' => 0,
            'max_score' => 20,
        ]);
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleHigh->getKey(),
            'min_score' => 21,
            'max_score' => 100,
        ]);
    }

    $postData = $this->questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                ? [$question->answers->random()->getKey()]
                : $question->answers->random()->getKey(),
        ];
    })->all();

    $response = $this->post(
        route('questionnaire.assessments.store', ['questionnaire' => $this->questionnaire->slug]),
        $postData,
    );

    $response->assertStatus(302);
});

it('fails validation when a wrong answer is posted that does not belong to the question', function () {
    config()->set('questionnaire.mode', 'default');

    $wrongQuestion = Question::factory()
        ->for($this->categories->random())
        ->for($this->questionnaire)
        ->has(Answer::factory()->count(5))
        ->create();

    $postData = $this->questionnaire->questions->mapWithKeys(function ($question) use ($wrongQuestion) {
        return [
            'questions_'.$wrongQuestion->getKey() => $wrongQuestion->type === InputTypeEnum::CHECKBOX
                ? [$wrongQuestion->answers->random()->getKey()]
                : $wrongQuestion->answers->random()->getKey(),
        ];
    })->all();

    $postData['_token'] = 'fake-csrf-token-for-testing';

    $response = $this->post(route('questionnaire.assessments.store', [
        'questionnaire' => $this->questionnaire->slug,
        ...$postData,
    ]));

    $response->assertStatus(302);
});

it('expects assessment in the database and stores answers (based on anonymous participant)', function () {
    config()->set('questionnaire.mode', 'default');

    $this->assertDatabaseCount('assessments', 0);

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $usedCategories = $this->questionnaire->questions->pluck('category_id')->unique();

    foreach ($usedCategories as $categoryId) {
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleLow->getKey(),
            'min_score' => 0,
            'max_score' => 20,
        ]);
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleHigh->id,
            'min_score' => 21,
            'max_score' => 100,
        ]);
    }

    $postDataCollection = $this->questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                ? [$question->answers->random()->getKey()]
                : $question->answers->random()->getKey(),
        ];
    });

    $firstQuestionKey = 'questions_'.$this->questionnaire->questions->first()->getKey();
    $expectedAnswerId = $postDataCollection[$firstQuestionKey];

    $response = $this->post(
        route('questionnaire.assessments.store', ['questionnaire' => $this->questionnaire->slug]),
        $postDataCollection->all(),
    );

    $response->assertStatus(302);

    $this->assertDatabaseCount('assessments', 1);

    $assessment = Assessment::latest()->first();

    expect($assessment)->not->toBeNull();

    // $this->assertEquals($this->participant->getKey(), $assessment->participant_id);
    $this->assertEquals($this->questionnaire->getKey(), $assessment->questionnaire_id);

    $this->assertDatabaseHas('assessment_answers', [
        'answer_id' => $expectedAnswerId,
        'assessment_id' => $assessment->getKey(),
    ]);
});

it('expects assessment in the database and stores answers (based on linked participant)', function () {
    config()->set('questionnaire.mode', 'default');

    $this->assertDatabaseCount('assessments', 0);

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $usedCategories = $this->questionnaire->questions->pluck('category_id')->unique();

    foreach ($usedCategories as $categoryId) {
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleLow->getKey(),
            'min_score' => 0,
            'max_score' => 20,
        ]);
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleHigh->id,
            'min_score' => 21,
            'max_score' => 100,
        ]);
    }

    $postDataCollection = $this->questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                ? [$question->answers->random()->getKey()]
                : $question->answers->random()->getKey(),
        ];
    });

    $firstQuestionKey = 'questions_'.$this->questionnaire->questions->first()->getKey();
    $expectedAnswerId = $postDataCollection[$firstQuestionKey];

    $sessionData = [
        'type' => $this->participant->getMorphClass(), // e.g., 'Concept7\LaravelQuestionnaire\Models\User'
        'id' => $this->participant->getKey(),       // e.g., 1
    ];

    $response = $this
        ->withSession([QuestionnaireContext::KEY => $sessionData])
        ->post(
            route('questionnaire.assessments.store', ['questionnaire' => $this->questionnaire->slug]),
            $postDataCollection->all(),
        );

    $response->assertStatus(302);

    $this->assertDatabaseCount('assessments', 1);

    $assessment = Assessment::latest()->first();

    expect($assessment)->not->toBeNull();

    $this->assertEquals($this->participant->getKey(), $assessment->assessmentable_id);

    $this->assertEquals($this->questionnaire->getKey(), $assessment->questionnaire_id);

    $this->assertDatabaseHas('assessment_answers', [
        'answer_id' => $expectedAnswerId,
        'assessment_id' => $assessment->getKey(),
    ]);

    $this->assertDatabaseHas('assessments', [
        'assessmentable_id' => $this->participant->getKey(),
        'assessmentable_type' => User::class,
    ]);
});

describe('assessment show template', function () {
    it('loads corresponding template', function ($calculationType, $assertViewIs) {
        config()->set('questionnaire.layout', 'laravel-questionnaire::layouts.app');
        // setup
        $questionnaire = Questionnaire::factory(['calculation_type' => $calculationType])
            ->has(
                Question::factory()
                    ->for($this->categories->random())
                    ->has(Answer::factory()
                        ->count(5))
            )
            ->create();

        $scaleLow = Scale::factory()->create(['name' => 'Low']);
        $scaleHigh = Scale::factory()->create(['name' => 'High']);

        $usedCategories = $questionnaire->questions->pluck('category_id')->unique();

        foreach ($usedCategories as $categoryId) {
            Interpretation::factory()->create([
                'category_id' => $categoryId,
                'scale_id' => $scaleLow->getKey(),
                'min_score' => 0,
                'max_score' => 20,
            ]);
            Interpretation::factory()->create([
                'category_id' => $categoryId,
                'scale_id' => $scaleHigh->id,
                'min_score' => 21,
                'max_score' => 100,
            ]);
        }

        $postDataCollection = $questionnaire->questions->mapWithKeys(function ($question) {
            return [
                'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                    ? [$question->answers->random()->getKey()]
                    : $question->answers->random()->getKey(),
            ];
        });

        $response = $this->post(
            route('questionnaire.assessments.store', ['questionnaire' => $questionnaire->slug]),
            $postDataCollection->all(),
        );

        $response->assertStatus(302);

        $assessment = Assessment::where('questionnaire_id', $questionnaire->getKey())
            ->latest()
            ->first();

        $response->assertRedirect(route('questionnaire.assessments.show', [
            'questionnaire' => $questionnaire->slug,
            'assessment' => $assessment->getKey(),
        ]));

        $response = $this->get(route('questionnaire.assessments.show', [
            'questionnaire' => $questionnaire->slug,
            'assessment' => $assessment->getKey(),
        ]));

        $response->assertStatus(200);

        $response->assertViewIs($assertViewIs);
    })->with([
        'it loads default template' => [
            'calculationType' => CalculationTypeEnum::AVERAGE,
            'assertViewIs' => 'laravel-questionnaire::assessments.results.average',
        ],
        'it loads sum template' => [
            'calculationType' => CalculationTypeEnum::SUM,
            'assertViewIs' => 'laravel-questionnaire::assessments.results.sum',
        ],
    ]);
});

describe('assessment create template', function () {
    it('loads corresponding template', function ($mode, $assertViewIs) {
        config()->set('questionnaire.mode', $mode);

        $response = $this->get(route('questionnaire.assessments.create', ['questionnaire' => $this->questionnaire->slug]));

        $response->assertViewIs($assertViewIs);
    })->with([
        'it loads default template' => ['mode' => 'default', 'assertViewIs' => 'laravel-questionnaire::assessments.create'],
        'it loads alpine wizard template' => ['mode' => 'alpine', 'assertViewIs' => 'laravel-questionnaire::assessments.wizard.create'],
        'it loads alpine_api wizard template' => ['mode' => 'alpine_via_api', 'assertViewIs' => 'laravel-questionnaire::assessments.wizard.create_api'],
    ]);
});

it('redirects to thank_you page instead of show route', function () {
    config()->set('questionnaire.thank_you_page_enabled', true);

    $questionnaire = Questionnaire::factory(['calculation_type' => CalculationTypeEnum::AVERAGE])
        ->has(
            Question::factory()
                ->for($this->categories->random())
                ->has(Answer::factory()
                    ->count(5))
        )
        ->create();

    $scaleLow = Scale::factory()->create(['name' => 'Low']);
    $scaleHigh = Scale::factory()->create(['name' => 'High']);

    $usedCategories = $questionnaire->questions->pluck('category_id')->unique();

    foreach ($usedCategories as $categoryId) {
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleLow->getKey(),
            'min_score' => 0,
            'max_score' => 20,
        ]);
        Interpretation::factory()->create([
            'category_id' => $categoryId,
            'scale_id' => $scaleHigh->id,
            'min_score' => 21,
            'max_score' => 100,
        ]);
    }

    $postDataCollection = $questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->type === InputTypeEnum::CHECKBOX
                ? [$question->answers->random()->getKey()]
                : $question->answers->random()->getKey(),
        ];
    });

    $response = $this->post(
        route('questionnaire.assessments.store', ['questionnaire' => $questionnaire->slug]),
        $postDataCollection->all(),
    );

    $response->assertRedirect(route('questionnaire.thankyou', $questionnaire->slug));
});
