<?php

use Concept7\LaravelQuestionnaire\Actions\Assessments\ProcessSubmittedAnswers;
use Concept7\LaravelQuestionnaire\Models\Answer;
use Concept7\LaravelQuestionnaire\Models\Assessment;
use Concept7\LaravelQuestionnaire\Models\Category;
use Concept7\LaravelQuestionnaire\Models\Question;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Concept7\LaravelQuestionnaire\Models\User;
use Illuminate\Support\Collection;

beforeEach(function () {
    config()->set('questionnaire.models.participant', User::class);
});

it('can parses submitted answers', function () {
    $categories = Category::factory()->count(5)->create();
    $questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($categories->random())
                ->has(Answer::factory()->count(5))
                ->count(5)
        )
        ->create();

    $rawAnswers = $questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->answers->random()->getKey(),
        ];
    });

    $participant = User::factory()->create();

    $payload = collect([
        'questionnaire' => $questionnaire,
        'assessment' => Assessment::factory()
            ->for($questionnaire)
            ->state([
                'assessmentable_type' => User::class,
                'assessmentable_id' => $participant->getKey(),
            ])
            ->create(),
        'assessmentAnswers' => $rawAnswers,
    ]);

    $pipe = new ProcessSubmittedAnswers;
    $next = fn ($payload) => $payload;

    $resultPayload = $pipe->handle($payload, $next);

    expect($resultPayload->has('assessmentAnswers'))->toBeTrue();
    expect($resultPayload->get('assessmentAnswers'))->toBeInstanceOf(Collection::class);
});

it('throws an exception if the assessment is missing', function () {
    $payload = collect([
        'assessmentAnswers' => ['questions_1' => 1], // Provide answers
        // But NO 'assessment' key
    ]);
    $pipe = new ProcessSubmittedAnswers;
    $next = fn ($payload) => $payload;

    // This test now expects the 'InvalidArgumentException' to be thrown
    expect(fn () => $pipe->handle($payload, $next))->toThrow(InvalidArgumentException::class);
});

test('save assessment answers', function () {
    $user = User::factory()->create();
    $categories = Category::factory()->count(5)->create();
    $questionnaire = Questionnaire::factory()
        ->has(
            Question::factory()
                ->for($categories->random())
                ->has(Answer::factory()->count(5))
                ->count(5)
        )
        ->create();

    $rawAnswers = $questionnaire->questions->mapWithKeys(function ($question) {
        return [
            'questions_'.$question->getKey() => $question->answers->random()->getKey(),
        ];
    });

    $payload = collect([
        'questionnaire' => $questionnaire,
        'participant' => $user,
        'assessment' => Assessment::factory()
            ->for($questionnaire)
            ->state([
                'assessmentable_id' => $user->getKey(),
                'assessmentable_type' => User::class,
            ])
            ->create(),
        'assessmentAnswers' => $rawAnswers,
    ]);

    $pipe = new ProcessSubmittedAnswers;
    $next = fn ($payload) => $payload;
    $pipe->handle($payload, $next);

    $this->assertDatabaseHas('assessment_answers', ['assessment_id' => 1]);
});
