<?php

use Concept7\LaravelQuestionnaire\Actions\Assessments\CalculateScores;
use Concept7\LaravelQuestionnaire\Enums\CalculationTypeEnum;
use Concept7\LaravelQuestionnaire\Models\Answer;
use Concept7\LaravelQuestionnaire\Models\Category;
use Concept7\LaravelQuestionnaire\Models\Question;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Illuminate\Support\Collection;

beforeEach(function () {
    config()->set('questionnaire.models.user', User::class);
    config()->set('questionnaire.models.category', Category::class);
    config()->set('questionnaire.models.question', Question::class);
    config()->set('questionnaire.models.answer', Answer::class);
    config()->set('questionnaire.models.questionnaire', Questionnaire::class);
});

it('calculates scores per category', function () {
    $questionnaire = Questionnaire::factory()->create();
    $category1 = Category::factory()->create();
    $category2 = Category::factory()->create();

    $question_category1 = Question::factory()->for($category1)->for($questionnaire)->create();
    $question_category2 = Question::factory()->for($category2)->for($questionnaire)->create();

    $answer1_cat1 = Answer::factory()->for($question_category1)->create(['score' => 10]);
    $answer2_cat1 = Answer::factory()->for($question_category1)->create(['score' => 20]);
    $answer3_cat2 = Answer::factory()->for($question_category2)->create(['score' => 50]);

    $submittedAnswers = collect([
        ['question_id' => $question_category1->getKey(), 'answer_id' => $answer1_cat1->getKey()],
        ['question_id' => $question_category1->getKey(), 'answer_id' => $answer2_cat1->getKey()],
        ['question_id' => $question_category1->getKey(), 'answer_id' => $answer3_cat2->getKey()],
    ]);

    // Create the initial payload
    $payload = collect([
        'questionnaire' => $questionnaire,
        'submittedAnswers' => $submittedAnswers,
    ]);

    $pipe = new CalculateScores;
    $next = fn ($payload) => $payload;

    $resultPayload = $pipe->handle($payload, $next);

    expect($resultPayload->has('scoresPerCategory'))->toBeTrue();

    $scores = $resultPayload->get('scoresPerCategory');

    // Category 1 should be 10 + 20 = 30
    expect($scores->get($category1->getKey()))->toBe(30);

    // Category 2 should be 50
    expect($scores->get($category2->getKey()))->toBe(50);
});

it('returns an empty collection when no answers are submitted', function () {
    // 1. Setup
    $questionnaire = Questionnaire::factory()->create(); // Default SUM type

    // 2. Define empty submitted answers
    $submittedAnswers = collect([]);

    $payload = collect([
        'questionnaire' => $questionnaire,
        'submittedAnswers' => $submittedAnswers,
    ]);

    // 3. Act
    $pipe = new CalculateScores;
    $next = fn ($payload) => $payload;
    $resultPayload = $pipe->handle($payload, $next);

    // 4. Assert
    expect($resultPayload->has('scoresPerCategory'))->toBeTrue();
    $scores = $resultPayload->get('scoresPerCategory');

    // The result should be an empty collection
    expect($scores)->toBeInstanceOf(Collection::class);
    expect($scores->isEmpty())->toBeTrue();
});

it('calculates SUM scores correctly with zero and negative values', function () {
    // 1. Setup
    $questionnaire = Questionnaire::factory()->create(); // Default SUM type
    $category1 = Category::factory()->create();
    $q1_cat1 = Question::factory()->for($category1)->for($questionnaire)->create();

    $a1 = Answer::factory()->for($q1_cat1)->create(['score' => 10]);
    $a2 = Answer::factory()->for($q1_cat1)->create(['score' => 0]);
    $a3 = Answer::factory()->for($q1_cat1)->create(['score' => -5]);

    // 2. Define submitted answers
    $submittedAnswers = collect([
        ['question_id' => $q1_cat1->getKey(), 'answer_id' => $a1->getKey()],
        ['question_id' => $q1_cat1->getKey(), 'answer_id' => $a2->getKey()],
        ['question_id' => $q1_cat1->getKey(), 'answer_id' => $a3->getKey()],
    ]);

    $payload = collect([
        'questionnaire' => $questionnaire,
        'submittedAnswers' => $submittedAnswers,
    ]);

    // 3. Act
    $pipe = new CalculateScores;
    $next = fn ($payload) => $payload;
    $resultPayload = $pipe->handle($payload, $next);

    // 4. Assert
    $scores = $resultPayload->get('scoresPerCategory');

    // Category 1 sum: (10 + 0 + (-5)) = 5
    expect($scores->get($category1->getKey()))->toBe(5);
});

it('calculates AVERAGE scores correctly with zero and negative values', function () {
    // 1. Setup
    $questionnaire = Questionnaire::factory()->create([
        'calculation_type' => CalculationTypeEnum::AVERAGE,
    ]);
    $category1 = Category::factory()->create();
    $q1_cat1 = Question::factory()->for($category1)->for($questionnaire)->create();

    $a1 = Answer::factory()->for($q1_cat1)->create(['score' => 10]);
    $a2 = Answer::factory()->for($q1_cat1)->create(['score' => 0]);
    $a3 = Answer::factory()->for($q1_cat1)->create(['score' => -5]);

    // 2. Define submitted answers
    $submittedAnswers = collect([
        ['question_id' => $q1_cat1->getKey(), 'answer_id' => $a1->getKey()],
        ['question_id' => $q1_cat1->getKey(), 'answer_id' => $a2->getKey()],
        ['question_id' => $q1_cat1->getKey(), 'answer_id' => $a3->getKey()],
    ]);

    $payload = collect([
        'questionnaire' => $questionnaire,
        'submittedAnswers' => $submittedAnswers,
    ]);

    // 3. Act
    $pipe = new CalculateScores;
    $next = fn ($payload) => $payload;
    $resultPayload = $pipe->handle($payload, $next);

    // 4. Assert
    $scores = $resultPayload->get('scoresPerCategory');

    // Category 1 average: (10 + 0 - 5) / 3 = 1.666...
    expect($scores->get($category1->getKey()))->toBe(5 / 3);
});

it('calculates score correctly for multiple different questions in the same category', function () {
    $questionnaire = Questionnaire::factory()->create(); // Default SUM
    $category1 = Category::factory()->create();

    // Question 1 (in Category 1)
    $q1_cat1 = Question::factory()->for($category1)->for($questionnaire)->create();
    $a1_q1 = Answer::factory()->for($q1_cat1)->create(['score' => 10]);

    // Question 2 (also in Category 1)
    $q2_cat1 = Question::factory()->for($category1)->for($questionnaire)->create();
    $a1_q2 = Answer::factory()->for($q2_cat1)->create(['score' => 30]);

    // 2. Define submitted answers
    $submittedAnswers = collect([
        // Answer from Question 1
        ['question_id' => $q1_cat1->getKey(), 'answer_id' => $a1_q1->getKey()],
        // Answer from Question 2
        ['question_id' => $q2_cat1->getKey(), 'answer_id' => $a1_q2->getKey()],
    ]);

    $payload = collect([
        'questionnaire' => $questionnaire,
        'submittedAnswers' => $submittedAnswers,
    ]);

    // 3. Act
    $pipe = new CalculateScores;
    $next = fn ($payload) => $payload;
    $resultPayload = $pipe->handle($payload, $next);

    // 4. Assert
    $scores = $resultPayload->get('scoresPerCategory');

    // Category 1 sum: (10 + 30) = 40
    expect($scores->get($category1->getKey()))->toBe(40);
});

it('calculates HIGHEST_SCORE correctly', function () {
    // 1. Setup
    $questionnaire = Questionnaire::factory()->create([
        'calculation_type' => CalculationTypeEnum::HIGHEST_SCORE,
    ]);
    $category1 = Category::factory()->create();
    $question = Question::factory()->for($category1)->for($questionnaire)->create();

    // Create answers with different scores
    $lowAnswer = Answer::factory()->for($question)->create(['score' => 2]);
    $highAnswer = Answer::factory()->for($question)->create(['score' => 7]);
    $midAnswer = Answer::factory()->for($question)->create(['score' => 5]);

    // 2. Simulate a multi-select submission (user picked all three)
    $submittedAnswers = collect([
        ['question_id' => $question->getKey(), 'answer_id' => $lowAnswer->getKey()],
        ['question_id' => $question->getKey(), 'answer_id' => $highAnswer->getKey()],
        ['question_id' => $question->getKey(), 'answer_id' => $midAnswer->getKey()],
    ]);

    $payload = collect([
        'questionnaire' => $questionnaire,
        'submittedAnswers' => $submittedAnswers,
    ]);

    // 3. Act
    $pipe = new CalculateScores;
    $resultPayload = $pipe->handle($payload, fn ($p) => $p);

    // 4. Assert
    $scores = $resultPayload->get('scoresPerCategory');

    // Should pick 7, ignoring 2 and 5
    expect($scores->get($category1->getKey()))->toBe(7);
});

it('calculates WEIGHTED_AVERAGE correctly with explicit weights', function () {
    // 1. Setup
    $questionnaire = Questionnaire::factory()->create([
        'calculation_type' => CalculationTypeEnum::WEIGHTED_AVERAGE,
    ]);
    $category = Category::factory()->create();

    // Question 1: Weight 2. User scores 5 (out of max 10).
    // Achieved: 5 * 2 = 10. Possible: 10 * 2 = 20.
    $q1 = Question::factory()->for($category)->for($questionnaire)->create(['weight' => 2]);
    Answer::factory()->for($q1)->create(['score' => 10]); // Max possible for q1
    $a1 = Answer::factory()->for($q1)->create(['score' => 5]); // User choice

    // Question 2: Weight 1 (Default). User scores 10 (out of max 10).
    // Achieved: 10 * 1 = 10. Possible: 10 * 1 = 10.
    $q2 = Question::factory()->for($category)->for($questionnaire)->create(['weight' => 1]);
    $a2 = Answer::factory()->for($q2)->create(['score' => 10]); // Max possible & User choice

    $submittedAnswers = collect([
        ['question_id' => $q1->getKey(), 'answer_id' => $a1->getKey()],
        ['question_id' => $q2->getKey(), 'answer_id' => $a2->getKey()],
    ]);

    $payload = collect(['questionnaire' => $questionnaire, 'submittedAnswers' => $submittedAnswers]);

    // 3. Act
    $pipe = new CalculateScores;
    $resultPayload = $pipe->handle($payload, fn ($p) => $p);

    // 4. Assert
    // Total Achieved: 10 (q1) + 10 (q2) = 20
    // Total Possible: 20 (q1) + 10 (q2) = 30
    // Calculation: (20 / 30) * 100 = 66.666... -> Rounded to 66.67

    $scores = $resultPayload->get('scoresPerCategory');
    expect($scores->get($category->getKey()))->toBe(66.67);
});

it('uses default max score of 10 in WEIGHTED_AVERAGE when no siblings exist', function () {
    $questionnaire = Questionnaire::factory()->create([
        'calculation_type' => CalculationTypeEnum::WEIGHTED_AVERAGE,
    ]);
    $category = Category::factory()->create();

    $q1 = Question::factory()->for($category)->for($questionnaire)->create();

    $a1 = Answer::factory()->for($q1)->create(['score' => 5]);

    $submittedAnswers = collect([
        ['question_id' => $q1->getKey(), 'answer_id' => $a1->getKey()],
    ]);

    $payload = collect(['questionnaire' => $questionnaire, 'submittedAnswers' => $submittedAnswers]);

    $pipe = new CalculateScores;
    $resultPayload = $pipe->handle($payload, fn ($p) => $p);

    // Achieved: 5. Max Possible (based on DB): 5. Result: 100%.
    $scores = $resultPayload->get('scoresPerCategory');
    expect($scores->get($category->getKey()))->toBe(100.0);
});

it('returns 0 for WEIGHTED_AVERAGE if total possible points is zero', function () {
    $questionnaire = Questionnaire::factory()->create([
        'calculation_type' => CalculationTypeEnum::WEIGHTED_AVERAGE,
    ]);
    $category = Category::factory()->create();

    $q1 = Question::factory()->for($category)->for($questionnaire)->create(['weight' => 0]);

    // Even if answer has score, weight 0 makes points 0.
    $a1 = Answer::factory()->for($q1)->create(['score' => 10]);

    $submittedAnswers = collect([
        ['question_id' => $q1->getKey(), 'answer_id' => $a1->getKey()],
    ]);

    $payload = collect(['questionnaire' => $questionnaire, 'submittedAnswers' => $submittedAnswers]);

    $pipe = new CalculateScores;
    $resultPayload = $pipe->handle($payload, fn ($p) => $p);

    $scores = $resultPayload->get('scoresPerCategory');

    // Should allow the flow to return 0 instead of throwing DivisionByZeroError
    expect($scores->get($category->getKey()))->toBe(0.0);
});

it('calculates scores separately for mixed categories in one submission', function () {
    // Setup: Default SUM type
    $questionnaire = Questionnaire::factory()->create();
    $catA = Category::factory()->create();
    $catB = Category::factory()->create();

    $qA = Question::factory()->for($catA)->for($questionnaire)->create();
    $qB = Question::factory()->for($catB)->for($questionnaire)->create();

    $ansA = Answer::factory()->for($qA)->create(['score' => 10]);
    $ansB = Answer::factory()->for($qB)->create(['score' => 20]);

    $submittedAnswers = collect([
        ['question_id' => $qA->getKey(), 'answer_id' => $ansA->getKey()],
        ['question_id' => $qB->getKey(), 'answer_id' => $ansB->getKey()],
    ]);

    $payload = collect(['questionnaire' => $questionnaire, 'submittedAnswers' => $submittedAnswers]);

    $pipe = new CalculateScores;
    $resultPayload = $pipe->handle($payload, fn ($p) => $p);

    $scores = $resultPayload->get('scoresPerCategory');

    // Ensure we have 2 distinct keys
    expect($scores->count())->toBe(2)
        ->and($scores->get($catA->getKey()))->toBe(10)
        ->and($scores->get($catB->getKey()))->toBe(20);
});
