<?php

use Concept7\LaravelQuestionnaire\Actions\Assessments\Invitations\SendQuestionnaireInvitation;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Concept7\LaravelQuestionnaire\Models\User;
use Concept7\LaravelQuestionnaire\Notifications\QuestionnaireInvitation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\URL;

beforeEach(function () {
    Config::set('questionnaire.models.user', User::class);
    Config::set('questionnaire.notifications.invitation', QuestionnaireInvitation::class);
    Config::set('questionnaire.models.questionnaire', Questionnaire::class);
});

it('sends a notification with a valid signed url via the action', function () {
    Notification::fake();

    $questionnaire = Questionnaire::factory()->create();
    $user = User::factory()->create();

    $action = new SendQuestionnaireInvitation;
    $action->execute($user, $questionnaire);

    Notification::assertSentTo(
        [$user],
        QuestionnaireInvitation::class,
        function ($notification, $channels) use ($user, $questionnaire) {
            $url = $notification->signedUrl;
            $isValid = URL::hasValidSignature(Request::create($url));

            $isCorrectRoute = str_contains($url, "/questionnaire/invite/{$questionnaire->slug}/{$user->id}");

            return $isValid && $isCorrectRoute;
        },
    );
});

it('sets the session and redirects when visiting the signed link', function () {
    $questionnaire = Questionnaire::factory()->create();
    $user = User::factory()->create();

    $url = URL::temporarySignedRoute('questionnaire.invite', now()->addHour(), [
        'questionnaire' => $questionnaire->slug,
        'user' => $user->getKey(),
    ]);

    // 1. Capture the response
    $response = $this->get($url);

    // 2. Assert Redirect
    $response->assertRedirect(
        route('questionnaire.assessments.create', [
            'questionnaire' => $questionnaire->slug,
        ]),
    );

    // 3. Assert Session (ON THE RESPONSE OBJECT)
    $response->assertSessionHas(
        "allowed_assessment_{$questionnaire->id}",
        $user->id,
    );
});

it('rejects an invalid or tampered signed link', function () {
    $questionnaire = Questionnaire::factory()->create();
    $user = User::factory()->create();

    $url = URL::temporarySignedRoute('questionnaire.invite', now()->addHour(), [
        'questionnaire' => $questionnaire->slug,
        'user' => $user->getKey(),
    ]);

    // Tamper with the URL (change user ID)
    $tamperedUrl = str_replace((string) $user->id, '99999', $url);

    $response = $this->get($tamperedUrl);
    $response->assertStatus(404);
});
