<?php

namespace Concept7\LaravelQuestionnaire\Http\Controllers\Questionnaire;

use Concept7\LaravelQuestionnaire\Actions\Assessments\CalculateScores;
use Concept7\LaravelQuestionnaire\Actions\Assessments\CreateAssessment;
use Concept7\LaravelQuestionnaire\Actions\Assessments\DetermineAndProcessResults;
use Concept7\LaravelQuestionnaire\Actions\Assessments\ProcessSubmittedAnswers;
use Concept7\LaravelQuestionnaire\Enums\CalculationTypeEnum;
use Concept7\LaravelQuestionnaire\Helpers\ChartHelper;
use Concept7\LaravelQuestionnaire\Http\Controllers\Controller;
use Concept7\LaravelQuestionnaire\Http\Requests\StoreAssessmentRequest;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Facades\Pipeline;
use Illuminate\Support\Facades\Request;

class AssessmentController extends Controller
{
    /**
     * Show the form for creating a new resource.
     *
     * @param  mixed  $questionnaire
     */
    public function create($questionnaire): View
    {
        $view = match (config('questionnaire.mode')) {
            'alpine' => 'laravel-questionnaire::assessments.wizard.create',
            'alpine_via_api' => 'laravel-questionnaire::assessments.wizard.create_api',
            default => 'laravel-questionnaire::assessments.create',
        };

        return view($view, [
            'questionnaire' => $questionnaire->loadMissing([
                'questions.answers',
                'questions.category.interpretation.scale',
            ]),
            'questions' => $questionnaire->questions,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  mixed  $questionnaire
     * @return mixed
     */
    public function store($questionnaire, StoreAssessmentRequest $request)
    {
        $respondent = $this->resolveRespondent($request);

        return Pipeline::send(
            collect([
                'questionnaire' => $questionnaire,
                'user' => $respondent,
                'assessmentAnswers' => $request->validated(),
            ]),
        )
            ->through([
                CreateAssessment::class,
                ProcessSubmittedAnswers::class,
                CalculateScores::class,
                DetermineAndProcessResults::class,
            ])
            ->then(function ($payload) use ($request) {
                if (
                    config('questionnaire.mode') === 'alpine_via_api' &&
                    $request->wantsJson()
                ) {
                    return response()->json([
                        'redirect_url' => route(
                            'questionnaire.assessments.show',
                            [
                                'questionnaire' => $payload->get('assessment')->questionnaire->slug,
                                'assessment' => $payload->get('assessment')->getKey(),
                            ],
                        ),
                    ]);
                }

                if (config('questionnaire.thank_you_page_enabled')) {
                    return redirect()->route('questionnaire.thankyou', ['questionnaire' => $payload->get('assessment')->questionnaire->slug]);
                }

                return redirect()->route('questionnaire.assessments.show', [
                    'questionnaire' => $payload->get('assessment')
                        ->questionnaire->slug,
                    'assessment' => $payload->get('assessment')->getKey(),
                ]);
            });
    }

    /**
     * Display the specified resource.
     *
     * @param  mixed  $questionnaire
     * @param  mixed  $assessment
     */
    public function show(Request $request, $questionnaire, $assessment): View
    {
        $templatePath = $this->findTemplatePath(questionnaire: $questionnaire);

        $templateSpecificData = match ($templatePath) {
            'laravel-questionnaire::assessments.results.sum' => [
                'chartData' => ChartHelper::buildChartDataFromAssessment(
                    $assessment,
                ),
                'chartOptions' => ChartHelper::getDefaultChartOptions(),
            ],
            default => [],
        };

        return view(
            $templatePath,
            array_merge(
                [
                    'questionnaire' => $questionnaire,
                    'assessment' => $assessment,
                ],
                $templateSpecificData,
            ),
        );
    }

    /**
     * @param  Questionnaire  $questionnaire
     */
    private function findTemplatePath($questionnaire): string
    {
        if (empty($templatePath)) {
            $templatePath = match ($questionnaire->calculation_type) {
                CalculationTypeEnum::AVERAGE => 'laravel-questionnaire::assessments.results.average',
                CalculationTypeEnum::SUM => 'laravel-questionnaire::assessments.results.sum',
                CalculationTypeEnum::HIGHEST_SCORE,
                CalculationTypeEnum::WEIGHTED_AVERAGE => 'laravel-questionnaire::assessments.results.sum',

                default => 'laravel-questionnaire::assessments.show',
            };
        }

        return $templatePath;
    }

    /**
     * @return mixed|null
     */
    public function resolveRespondent(StoreAssessmentRequest $request): mixed
    {
        if (auth()->guard()->check()) {
            return auth()->guard()->user();
        }

        // auth stuff is getting extracted from laravel config
        if ($request->respondent_id) {
            $userProvider = config('auth.guards.web.provider');
            $userClass = config("auth.providers.{$userProvider}.model");

            return $userClass::find($request->respondent_id);
        }

        return null;
    }

    public function thankYouPage(Request $request, $questionnaire): View
    {
        return view('questionnaire::assessments.thank_you');
    }
}
