<?php

namespace Concept7\LaravelQuestionnaire\Http\Requests;

use Concept7\LaravelQuestionnaire\Enums\InputTypeEnum;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class StoreAssessmentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        if (! $this->participant_id && ! config('questionnaire.allow_public_access')) {
            return false;
        }

        return true;
    }

    protected function prepareForValidation(): void
    {
        $routeParam = $this->route('participant');

        if ($routeParam) {
            $value = ($routeParam instanceof \Illuminate\Database\Eloquent\Model)
                ? $routeParam->getRouteKey()
                : $routeParam;

            $this->merge(['participant_id' => $value]);
        }
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $questionnaire = $this->route('questionnaire');
        $questionnaire->load(['questions.answers']);
        $rules = [];

        if ($this->participant_id) {
            $modelClass = config('questionnaire.models.participant');

            if (! $modelClass) {
                throw new \Exception("Configuration key 'questionnaire.models.participant' is not set.");
            }

            $instance = new $modelClass;
            $column = is_numeric($this->participant_id)
                ? $instance->getKeyName()
                : $instance->getRouteKeyName();

            $rules['participant_id'] = [
                'required',
                "exists:{$instance->getTable()},{$column}",
            ];
        }

        foreach ($questionnaire->questions as $question) {
            $key = 'questions_'.$question->id;
            $validOptionIds = $question->answers->pluck('id')->all();

            switch ($question->type) {
                case InputTypeEnum::RADIO:
                case InputTypeEnum::SELECT:
                case InputTypeEnum::SLIDER:
                    $rules[$key] = [
                        'required',
                        'integer',
                        Rule::in($validOptionIds),
                    ];
                    break;

                case InputTypeEnum::CHECKBOX:
                    $rules[$key] = ['required', 'array'];
                    $rules[$key.'.*'] = [
                        'required',
                        'integer',
                        Rule::in($validOptionIds),
                    ];
                    break;
            }
        }

        return $rules;
    }
}
