<?php

namespace Concept7\LaravelQuestionnaire\Http\Controllers\Questionnaire;

use Concept7\LaravelQuestionnaire\Actions\Assessments\CalculateScores;
use Concept7\LaravelQuestionnaire\Actions\Assessments\CreateAssessment;
use Concept7\LaravelQuestionnaire\Actions\Assessments\DetermineAndProcessResults;
use Concept7\LaravelQuestionnaire\Actions\Assessments\GetParticipant;
use Concept7\LaravelQuestionnaire\Actions\Assessments\ProcessSubmittedAnswers;
use Concept7\LaravelQuestionnaire\Enums\CalculationTypeEnum;
use Concept7\LaravelQuestionnaire\Helpers\ChartHelper;
use Concept7\LaravelQuestionnaire\Http\Controllers\Controller;
use Concept7\LaravelQuestionnaire\Http\Requests\StoreAssessmentRequest;
use Concept7\LaravelQuestionnaire\Models\Questionnaire;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Facades\Pipeline;
use Illuminate\Support\Facades\Request;

class AssessmentController extends Controller
{
    /**
     * Show the form for creating a new resource.
     *
     * @param  mixed  $questionnaire
     * @param  mixed  $participantId
     */
    public function create($questionnaire, $participantIdentifier = null): View
    {
        if (! config('questionnaire.allow_public_access') && ! $participantIdentifier) {
            abort(403);
        }

        $participantClass = config('questionnaire.models.participant');

        $participant = $participantIdentifier ? $participantClass::query()
            ->where(config('questionnaire.participant_route_key', 'id'), $participantIdentifier)->first() : null;

        $view = match (config('questionnaire.mode')) {
            'alpine' => 'laravel-questionnaire::assessments.wizard.create',
            'alpine_via_api' => 'laravel-questionnaire::assessments.wizard.create_api',
            default => 'laravel-questionnaire::assessments.create',
        };

        return view($view, [
            'questionnaire' => $questionnaire->loadMissing([
                'questions.answers',
                'questions.category.interpretation.scale',
            ]),
            'questions' => $questionnaire->questions,
            'participant' => $participant,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  mixed  $questionnaire
     * @param  mixed  $participantId
     * @return mixed
     */
    public function store($questionnaire, StoreAssessmentRequest $request)
    {
        return Pipeline::send(
            collect([
                'questionnaire' => $questionnaire,
                'validatedRequest' => $request->validated(),
            ]),
        )
            ->through([
                GetParticipant::class,
                CreateAssessment::class,
                ProcessSubmittedAnswers::class,
                CalculateScores::class,
                DetermineAndProcessResults::class,
            ])
            ->then(function ($payload) use ($request) {

                if (config('questionnaire.thank_you_page_enabled')) {
                    return redirect()->route('questionnaire.thankyou', [
                        'questionnaire' => $payload->get('assessment')->questionnaire->slug,
                    ]);
                }

                if ($payload->has('participant')) {
                    $participant = $payload->get('participant');
                } else {
                    throw new \Exception(__('Participant not found'));
                }

                if (
                    config('questionnaire.mode') === 'alpine_via_api' &&
                    $request->wantsJson()
                ) {
                    return response()->json([
                        'redirect_url' => route(
                            'questionnaire.assessments.show',
                            [
                                'questionnaire' => $payload->get('assessment')->questionnaire->slug,
                                'assessment' => $payload->get('assessment')->getKey(),
                                'participant' => $participant?->getKey(),
                            ],
                        ),
                    ]);
                }

                return redirect()->route('questionnaire.assessments.show', [
                    'questionnaire' => $payload->get('assessment')->questionnaire->slug,
                    'assessment' => $payload->get('assessment')->getKey(),
                    'participant' => $participant?->getKey() ?? null,
                ]);
            });
    }

    /**
     * Display the specified resource.
     *
     * @param  mixed  $questionnaire
     * @param  mixed  $assessment
     */
    public function show(Request $request, Questionnaire $questionnaire, $assessment)
    {
        $templatePath = $this->findTemplatePath(questionnaire: $questionnaire);

        $templateSpecificData = match ($templatePath) {
            'laravel-questionnaire::assessments.results.sum' => [
                'chartData' => ChartHelper::buildChartDataFromAssessment(
                    $assessment,
                ),
                'chartOptions' => ChartHelper::getDefaultChartOptions(),
            ],
            default => [],
        };

        return view(
            $templatePath,
            array_merge(
                [
                    'questionnaire' => $questionnaire,
                    'assessment' => $assessment,
                ],
                $templateSpecificData,
            ),
        );
    }

    /**
     * @param  Questionnaire  $questionnaire
     */
    private function findTemplatePath($questionnaire): string
    {
        if (empty($templatePath)) {
            $templatePath = match ($questionnaire->calculation_type) {
                CalculationTypeEnum::AVERAGE => 'laravel-questionnaire::assessments.results.average',
                CalculationTypeEnum::SUM => 'laravel-questionnaire::assessments.results.sum',
                CalculationTypeEnum::HIGHEST_SCORE,
                CalculationTypeEnum::WEIGHTED_AVERAGE => 'laravel-questionnaire::assessments.results.sum',

                default => 'laravel-questionnaire::assessments.show',
            };
        }

        return $templatePath;
    }

    /**
     * @param  mixed  $questionnaire
     */
    public function thankYou(Request $request, $questionnaire): View
    {
        return view('laravel-questionnaire::assessments.thank_you');
    }
}
