@extends(config('questionnaire.layout', 'layouts.app'))

@section('content')
    <div
        x-data="app()"
        x-cloak
    >
        <form
            method="POST"
            action="{{ route('questionnaire.assessments.store', ['questionnaire' => $questionnaire->slug]) }}"
        >
            @csrf
            <div class="max-w-3xl mx-auto px-4 py-10">
                <div x-show="step <= totalQuestions && totalQuestions > 0">
                    <div class="uppercase tracking-wide text-xs font-bold text-gray-500 mb-1 leading-tight"
                         x-text="`{{ __("Step") }}: ${step} {{ __("of") }} ${totalQuestions}`"
                    ></div>
                    @if( config('questionnaire.with_progress_bar') )
                        <div
                            class="progress-bar bg-blue-500 h-2.5 rounded-full mb-4"
                            :style="{ width: progressPercentage + '%'  }"
                        ></div>
                    @endif

                    <div class="flex flex-col md:flex-row md:items-center md:justify-between">
                        <div class="flex-1">
                            <template
                                x-for="(question, index) in questions"
                                :key="question.id"
                            >
                                <div
                                    x-show="step === (index + 1)"
                                    x-transition:enter="transition ease-out duration-300"
                                    x-transition:enter-start="opacity-0 transform -translate-x-4"
                                    x-transition:enter-end="opacity-100 transform translate-x-0"
                                    x-transition:leave="transition ease-in duration-200"
                                    x-transition:leave-start="opacity-100 transform translate-x-0"
                                    x-transition:leave-end="opacity-0 transform translate-x-4"
                                >
                                    <h3
                                        class="text-2xl font-semibold text-gray-900 mb-6"
                                        x-html="question.text"
                                    >
                                    </h3>
                                    <div class="flex flex-col gap-y-3">
                                        <template
                                            x-for="answer in question.answers"
                                            :key="answer.id"
                                        >
                                            <div>
                                                <input
                                                    type="radio"
                                                    x-model="form['questions_' + question.id]"
                                                    class="sr-only peer"
                                                    :id="`answer_${answer.id}`"
                                                    :name="`questions_${question.id}`"
                                                    :value="answer.id"
                                                    @change="form.validate(`questions_${question.id}`)"
                                                >
                                                <label
                                                    :for="`answer_${answer.id}`"
                                                    class="block w-full cursor-pointer text-left p-4 bg-white border border-gray-300 rounded-lg shadow-sm
                                                           hover:bg-gray-50 hover:border-blue-500
                                                           focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50
                                                           transition duration-150 ease-in-out
                                                           peer-checked:border-blue-600 peer-checked:ring-2 peer-checked:ring-blue-500 peer-checked:bg-blue-50"
                                                >
                                                    <span class="text-lg text-gray-700" x-text="answer.text"></span>
                                                </label>
                                            </div>
                                        </template>

                                        <template x-if="form.invalid('questions_' + question.id)">
                                            <div
                                                x-text="form.errors['questions_' + question.id]"
                                                class="text-red-500"
                                            ></div>
                                        </template>

                                        <button
                                            type="button"
                                            class="w-full bg-blue-600 text-white font-bold py-3 px-6 rounded-lg shadow-lg
                                               hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500
                                               focus:ring-opacity-50 transition duration-150 ease-in-out"
                                            @click="next()"
                                            :disabled="form.processing || loading"
                                            x-show="!isLastQuestion"
                                        >
                                            <span x-show="loading">Loading</span>
                                            <span x-show="!loading">
                                                {{ __("Next") }}
                                            </span>
                                        </button>

                                    </div>
                                </div>
                            </template>
                        </div>
                    </div>

                    <div class="mt-8" x-show="isLastQuestion">
                        <button
                            :disabled="form.processing"
                            class="w-full bg-blue-600 text-white font-bold py-3 px-6 rounded-lg shadow-lg
                                   hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500
                                   focus:ring-opacity-50 transition duration-150 ease-in-out"
                        >
                            {{ __("Submit Answers") }}
                        </button>
                    </div>
                </div>

                <div x-show="totalQuestions === 0">
                    <div class="text-gray-500">{{__("Loading question")}}...</div>
                </div>
            </div>
        </form>
    </div>
@endsection
@push('scripts')
    <script>
        function app() {
            return {
                step: window.Alpine.$persist(1),
                questions: @json($questions),
                form:null,
                loading:false,

                givenAnswers: window.Alpine.$persist({}).as('questionnaire_answers'),

                init(){
                  const initialAnswers = {};
                  this.questions.forEach(question => {
                      initialAnswers[`questions_${question.id}`] = '';
                  });

                const allAnswers = Object.assign(initialAnswers, this.givenAnswers);

                 this.form = this.$form(
                      'post',
                     '{{ route("questionnaire.assessments.store", ['questionnaire' => $questionnaire->slug])  }}',
                      allAnswers,
                  )
                },

                get totalQuestions() {
                    return this.questions ? this.questions.length : 0;
                },

                get isLastQuestion()
                {
                    return this.step === this.totalQuestions;
                },

                get progressPercentage(){
                    if(this.totalQuestions === 0){
                        return 0;
                    }
                    return (this.step / this.totalQuestions) * 100;
                },

                next(){
                    if(this.form.processing) return;

                    this.loading = true;
                    const currentQuestionId = this.questions[this.step -1].id;
                    const fieldName = `questions_${currentQuestionId}`;

                    this.form.validate({
                        only: [fieldName],
                        onSuccess: (response) => {
                            this.givenAnswers[fieldName] = this.form[fieldName];

                            if(!this.isLastQuestion){
                                this.step++
                            }
                            this.loading = false;
                        },
                        onValidationError: (error) => {
                            console.log(error);
                            this.loading = false;
                        }
                    });
                },
            }
        }
    </script>
@endpush
