<?php

namespace Concept7\Deployer\Recipe;

use Concept7\Deployer\Enum\HostingType;
use Concept7\Deployer\Traits\Environments;
use Concept7\Deployer\Traits\FilterTasks;
use Concept7\Deployer\Traits\Opcache;
use Concept7\Deployer\Traits\PhpRestart;
use Concept7\Deployer\Traits\PhpVersion;

use function Deployer\after;
use function Deployer\currentHost;
use function Deployer\desc;
use function Deployer\get;
use function Deployer\host;
use function Deployer\run;
use function Deployer\set;
use function Deployer\task;
use function Deployer\upload;
use function Deployer\which;

abstract class Base
{
    use Environments;
    use FilterTasks;
    use Opcache;
    use PhpRestart;
    use PhpVersion;

    protected string $repository;

    /** @var array<string> */
    protected array $deploySteps = ['deploy'];

    /** @var array<string> */
    protected array $deployerRecipes = [
        'recipe/common.php',
        'contrib/rsync.php',
        'contrib/cachetool.php',
        'contrib/php-fpm.php',
    ];

    /** @var array<string> */
    protected array $additionalDeployerRecipes = [];

    protected string $rootDir;

    protected bool $hasDotEnv = false;

    /** @var array<string> */
    protected array $defaultSharedFiles = [];

    /** @var array<string> */
    protected array $defaultSharedDirectories = [];

    /** @var array<string> */
    protected array $defaultWritableDirectories = [];

    protected int $keepReleases = 10;

    public function __construct()
    {
        foreach ($this->deployerRecipes as $deployerRecipe) {
            require_once $deployerRecipe;
        }

        foreach ($this->additionalDeployerRecipes as $additionalDeployerRecipe) {
            require_once $additionalDeployerRecipe;
        }

        $this->rootDir = $this->rootDirectory();
    }

    public function setRepository(string $repository): self
    {
        $this->repository = $repository;

        set('repository', $repository);

        return $this;
    }

    public function setApplication(string $name): self
    {
        set('application', $name);

        return $this;
    }

    /** @param  array<string>  $directories */
    public function setSharedDirectories(array $directories = []): self
    {
        if ($directories !== []) {
            $this->defaultSharedDirectories = $directories;
        }

        return $this;
    }

    /** @param  array<string>  $files */
    public function addSharedFiles(array $files = []): self
    {
        $this->defaultSharedFiles = [...$this->defaultSharedFiles, ...$files];

        return $this;
    }

    private function setSharedFiles(): self
    {
        $sharedFiles = $this->defaultSharedFiles;

        if ($this->hasDotEnv) {
            $sharedFiles = array_filter($sharedFiles, static function ($file) {
                return $file !== '.env';
            });
        }

        set('shared_files', $sharedFiles);

        return $this;
    }

    /** @param  array<string>  $directories */
    public function addSharedDirectories(array $directories = []): self
    {
        $this->defaultSharedDirectories = [...$this->defaultSharedDirectories, ...$directories];

        return $this;
    }

    private function setSharedDirs(): self
    {
        set('shared_dirs', $this->defaultSharedDirectories);

        return $this;
    }

    /** @param  array<string>  $directories */
    public function addWritableDirs(array $directories = []): self
    {
        $this->defaultWritableDirectories = [...$this->defaultWritableDirectories, ...$directories];

        return $this;
    }

    private function setWritableDirs(): self
    {
        set('writable_dirs', $this->defaultWritableDirectories);

        return $this;
    }

    public function setRsyncDirectories(string $src, string $dest): self
    {
        set('rsync_src', $src);
        set('rsync_dest', $dest);

        return $this;
    }

    public function rootDirectory(): string
    {
        // Change the second parameter to suit your needs
        return dirname(__FILE__, 6);
    }

    public function execute(): void
    {
        $this->setupEnvironments();
        $this->setupDefaults();
        $this->setupDotEnv();
        $this->setSharedDirs();
        $this->setSharedFiles();
        $this->setWritableDirs();
        $this->setupGit();
        $this->setupPhp();
        $this->setupComposer();
        $this->setupCopyFrontendAssets();
        $this->opcacheClear();
        $this->phpRestart();
        $this->setupDeploySteps();
        $this->loadExtraDeployerSteps();
    }

    private function setupEnvironments(): void
    {
        $this->setReviewApp();

        foreach ($this->environments as $environment => $data) {
            /** @var HostingType */
            $hosting_type = $data['hosting_type'];

            host($data['host'])
                ->setLabels([
                    'env' => $environment,
                    'hostingType' => $hosting_type->value,
                ])
                ->setHostname($data['hostname'])
                ->setPort($data['port'])
                ->setDeployPath($data['deploy_path'])
                ->setRemoteUser($data['remote_user'])
                ->setSshArguments(['-q -o SendEnv=NOMOTD'])
                ->set('worker_names', $data['worker_names'])
                ->set('branch', $data['branch'])
                ->set('keep_releases', $environment === 'production' ? $this->keepReleases : 2);
        }
    }

    private function setupDotEnv(): void
    {
        $env = getenv('ENV');

        if ($env) {
            $this->hasDotEnv = true;

            $sharedFiles = array_filter(get('shared_files', []), static function ($file) {
                return $file !== '.env';
            });

            set('shared_files', $sharedFiles);

            task(
                'deploy:secrets',
                static function () {
                    upload(getenv('ENV'), '{{release_path}}/.env');
                }
            );
            after('deploy:update_code', 'deploy:secrets');
        }
    }

    private function setupDefaults(): void
    {
        set('writable_mode', 'chmod');
        set('ssh_multiplexing', false);
        set('forward_agent', false);
        set('remote_user', 'forge');
        set('release_name', static function () {
            return (string) run('date +"%Y%m%d%H%M%S"');
        });
    }

    private function setupPhp(): void
    {
        set('bin/php', function () {
            return which(
                (currentHost()->get('labels')['env'] === 'production')
                ? 'php'
                : 'php'.$this->getPhpVersion()
            );
        });
    }

    private function setupComposer(): void
    {
        set('bin/composer', static function () {
            return '{{bin/php}} '.which('composer');
        });
    }

    private function setupGit(): void
    {
        set('bin/git', static function () {
            return which('git');
        });
    }

    public function setupCopyFrontendAssets(): void
    {
        set('rsync_src', $this->rootDir.'/public/build');
        set('rsync_dest', '{{release_path}}/public/build');
    }

    private function setupDeploySteps(): void
    {
        desc('Deploys your project');

        $deploySteps = $this->filterBasedOnInstalledComposerPackages($this->deploySteps);

        task('deploy', $deploySteps);

        after('deploy:failed', 'deploy:unlock');
    }

    /** @param  array<string>  $steps */
    public function setDeploySteps(array $steps): void
    {
        $this->deploySteps = $steps;
    }

    public function loadExtraDeployerSteps(): void {}
}
