<?php

namespace Concept7\Deployer\Traits;

use Concept7\Deployer\Enum\HostingType;

use function Deployer\currentHost;
use function Deployer\info;
use function Deployer\run;
use function Deployer\set;
use function Deployer\task;
use function Deployer\which;

trait Crontab
{
    /** @var array<int, array<string, string>> */
    private array $commands = [];

    public function addLaravelSchedulerToCrontab(): void
    {
        $this->addCustomCommandToCrontab('* * * * *', 'cd {{current_path}} && php artisan schedule:run >> /dev/null 2>&1');
    }

    public function addCustomCommandToCrontab(string $interval, string $command): void
    {
        $this->addToCrontab([$interval => $command]);
        $this->processCommands();
    }

    /** @param array<string, string> $command */
    private function addToCrontab(array $command): void
    {
        $this->commands[] = $command;
    }

    private function processCommands(): void
    {
        if (count($this->commands) === 0) {
            return;
        }

        set('bin/crontab', function () {
            $hostingType = $this->getHostingType();

            return match ($hostingType) {
                HostingType::LinQhost => '/usr/bin/crontab',
                HostingType::Combell => '/usr/local/bin/crontab',
                default => which('crontab')
            };
        });

        task('crontabs:update', function () {
            $hostingType = $this->getHostingType();

            foreach ($this->commands as $commandWithInterval) {
                if ($hostingType !== HostingType::Forge) {
                    foreach ($commandWithInterval as $interval => $command) {
                        $this->addCommandToCrontab($interval, $command);
                    }
                }
            }
        });
    }

    private function commandExists(string $interval, string $command): bool
    {
        $exists = 'crontab rule exists';
        $missing = 'crontab rule missing';
        $combined = $interval.' '.$command;
        $output = run(sprintf("{{bin/crontab}} -l | grep -q '%s' && echo '%s' || echo '%s'", $combined, $exists, $missing));

        return str_contains($output, $exists);
    }

    private function addCommandToCrontab(string $interval, string $command): void
    {
        if (! $this->commandExists($interval, $command)) {
            $combined = $interval.' '.$command;
            run('{{bin/crontab}} -l | { cat; echo "'.$combined.'"; } | {{bin/crontab}} -');
            info('Crontab rule added');
        }
    }

    private function getHostingType(): HostingType
    {
        return HostingType::tryFrom(currentHost()->get('labels')['hostingType']);
    }
}
