<?php

namespace Concept7\Deployer;

use Concept7\Deployer\Enum\HostingType;
use Concept7\Deployer\Enum\Recipe;
use Concept7\Deployer\Recipe\Base;
use Concept7\Deployer\Recipe\Laravel\Laravel;
use Concept7\Deployer\Recipe\Laravel\LaravelStatamic;
use Concept7\Deployer\Recipe\Laravel\Multitenancy\Laravel as MultitenancyLaravel;
use Concept7\Deployer\Recipe\Laravel\Multitenancy\LaravelStatamic as MultitenancyLaravelStatamic;
use Concept7\Deployer\Recipe\Laravel\Multitenancy\Statamic as MultitenancyStatamic;
use Concept7\Deployer\Recipe\Laravel\Statamic;
use Deployer\Exception\Exception;

class Loader
{
    private Recipe $recipe;

    private string $name;

    private string $repository;

    private float $phpVersion = 8.0;

    /** @var array<array<string>> */
    private array $environments = [];

    public function setProject(string $name, string $repository): self
    {
        $this->name = $name;
        $this->repository = $repository;

        return $this;
    }

    public function setRecipe(Recipe $recipe): self
    {
        $this->recipe = $recipe;

        return $this;
    }

    public function setPhpVersion(float $version): self
    {
        $this->phpVersion = $version;

        return $this;
    }

    public function setEnvironment(
        string $environment,
        string $host,
        string $deploy_path,
        string $hostname,
        string $remote_user,
        int $port = 22,
        string $branch = 'main',
        HostingType $hosting_type = HostingType::Forge,
        string $queue_worker_name = null,
        string $websocket_worker_name = null,
    ): self {
        if (isset($this->environments[$environment])) {
            throw new Exception('Environment: ['.$environment.'] already defined');
        }

        $this->environments[$environment] = [
            'host' => $host,
            'hostname' => $hostname,
            'remote_user' => $remote_user,
            'port' => $port,
            'branch' => $branch,
            'deploy_path' => $deploy_path,
            'hosting_type' => $hosting_type,
            'queue_worker_name' => $queue_worker_name,
            'websocket_worker_name' => $websocket_worker_name,
        ];

        return $this;
    }

    public function setStaging(
        string $host,
        string $deploy_path,
        string $hostname = '37.97.131.190',
        string $remote_user = 'forge',
        int $port = 22,
        string $branch = 'develop',
        HostingType $hosting_type = HostingType::Forge,
        string $queue_worker_name = null,
        string $websocket_worker_name = null,
    ): self {
        return $this->setEnvironment(
            environment: 'staging',
            host: $host,
            hostname: $hostname,
            remote_user: $remote_user,
            port: $port,
            branch: $branch,
            deploy_path: $deploy_path,
            hosting_type: $hosting_type,
            queue_worker_name: $queue_worker_name,
            websocket_worker_name: $websocket_worker_name,
        );
    }

    public function setAcceptance(
        string $host,
        string $deploy_path,
        string $hostname = '149.210.205.153',
        string $remote_user = 'forge',
        int $port = 22,
        string $branch = 'main',
        HostingType $hosting_type = HostingType::Forge,
        string $queue_worker_name = null,
        string $websocket_worker_name = null,
    ): self {
        return $this->setEnvironment(
            environment: 'acceptance',
            host: $host,
            hostname: $hostname,
            remote_user: $remote_user,
            port: $port,
            branch: $branch,
            deploy_path: $deploy_path,
            hosting_type: $hosting_type,
            queue_worker_name: $queue_worker_name,
            websocket_worker_name: $websocket_worker_name,
        );
    }

    public function setProduction(
        string $host,
        string $deploy_path,
        string $hostname,
        string $remote_user,
        int $port = 22,
        string $branch = 'main',
        HostingType $hosting_type = HostingType::LinQhost,
        string $queue_worker_name = null,
        string $websocket_worker_name = null,
    ): self {
        return $this->setEnvironment(
            environment: 'production',
            host: $host,
            hostname: $hostname,
            remote_user: $remote_user,
            port: $port,
            branch: $branch,
            deploy_path: $deploy_path,
            hosting_type: $hosting_type,
            queue_worker_name: $queue_worker_name,
            websocket_worker_name: $websocket_worker_name,
        );
    }

    public function execute(): void
    {
        $recipe = match ($this->recipe) {
            Recipe::Laravel => Laravel::class,
            Recipe::LaravelStatamic => LaravelStatamic::class,
            Recipe::Statamic => Statamic::class,
            // Recipe::MultitenancyLaravel => MultitenancyLaravel::class,
            Recipe::MultitenancyLaravelStatamic => MultitenancyLaravelStatamic::class,
            // Recipe::MultitenancyStatamic => MultitenancyStatamic::class,
            default => throw new \Exception('Recipe ['.$this->recipe->value.'] not found')
        };

        /** @var Base */
        $class = new $recipe();

        $class
            ->setRepository($this->repository)
            ->setApplication($this->name)
            ->setPhpVersion($this->phpVersion)
            ->setEnvironments($this->environments)
            ->run();
    }
}
