<?php

namespace Concept7\Deployer\Traits;

use Concept7\Deployer\Enum\HostingType;
use Exception;

trait Environments
{
    /** @var array<array<string>> */
    protected array $environments = [];

    /** @param  array<array<string>>  $environments */
    public function setEnvironments(array $environments): self
    {
        $this->environments = $environments;

        return $this;
    }

    public function setEnvironment(
        string $environment,
        string $host,
        string $deploy_path,
        string $hostname,
        string $remote_user,
        int $port = 22,
        string $branch = 'main',
        HostingType $hosting_type = HostingType::Forge,
        ?string $worker_names = null
    ): self {
        if (isset($this->environments[$environment])) {
            throw new Exception('Environment: ['.$environment.'] already defined');
        }

        $this->environments[$environment] = [
            'host' => $host,
            'hostname' => $hostname,
            'remote_user' => $remote_user,
            'port' => $port,
            'branch' => $branch,
            'deploy_path' => $deploy_path,
            'hosting_type' => $hosting_type,
            'worker_names' => $worker_names,
        ];

        return $this;
    }

    public function setStaging(
        string $host,
        string $deploy_path,
        string $hostname = 'staging.concept7.dev',
        string $remote_user = 'forge',
        int $port = 22,
        string $branch = 'develop',
        HostingType $hosting_type = HostingType::Forge,
        ?string $worker_names = null
    ): self {
        return $this->setEnvironment(
            environment: 'staging',
            host: $host,
            hostname: $hostname,
            remote_user: $remote_user,
            port: $port,
            branch: $branch,
            deploy_path: $deploy_path,
            hosting_type: $hosting_type,
            worker_names: $worker_names,
        );
    }

    public function setAcceptance(
        string $host,
        string $deploy_path,
        string $hostname = 'acceptance.concept7.dev',
        string $remote_user = 'forge',
        int $port = 22,
        string $branch = 'main',
        HostingType $hosting_type = HostingType::Forge,
        ?string $worker_names = null
    ): self {
        return $this->setEnvironment(
            environment: 'acceptance',
            host: $host,
            hostname: $hostname,
            remote_user: $remote_user,
            port: $port,
            branch: $branch,
            deploy_path: $deploy_path,
            hosting_type: $hosting_type,
            worker_names: $worker_names,
        );
    }

    public function setProduction(
        string $host,
        string $deploy_path,
        string $hostname,
        string $remote_user,
        int $port = 22,
        string $branch = 'main',
        HostingType $hosting_type = HostingType::LinQhost,
        ?string $worker_names = null
    ): self {
        return $this->setEnvironment(
            environment: 'production',
            host: $host,
            hostname: $hostname,
            remote_user: $remote_user,
            port: $port,
            branch: $branch,
            deploy_path: $deploy_path,
            hosting_type: $hosting_type,
            worker_names: $worker_names,
        );
    }

    public function setReviewApp(
        string $hostname = 'staging.concept7.dev',
        string $remote_user = 'forge',
        int $port = 22,
        HostingType $hosting_type = HostingType::Forge,
    ): self {
        $forgeVariables = collect($_SERVER)
            ->filter(static fn ($value, $key) => str_starts_with($key, 'DEPLOYER__'));

        if (! $forgeVariables->has([
            'DEPLOYER__SITE_URL',
            'DEPLOYER__REVIEW_APP_BRANCH_NAME',
            'DEPLOYER__FORGE_API_KEY',
            'DEPLOYER__FORGE_STAGING_SERVER_ID',
        ])) {
            return $this;
        }

        $host = $forgeVariables->get('DEPLOYER__SITE_URL');
        $deploy_path = '/home/forge/'.$forgeVariables->get('DEPLOYER__SITE_URL');
        $branch = $forgeVariables->get('DEPLOYER__REVIEW_APP_BRANCH_NAME');

        return $this->setEnvironment(
            environment: 'review-app',
            host: $host,
            hostname: $hostname,
            remote_user: $remote_user,
            port: $port,
            branch: $branch,
            deploy_path: $deploy_path,
            hosting_type: $hosting_type,
        );
    }
}
