<?php

namespace Concept7\Deployer\Traits;

use Concept7\Deployer\Enum\HostingType;

use function Deployer\currentHost;
use function Deployer\info;
use function Deployer\run;
use function Deployer\set;
use function Deployer\task;
use function Deployer\which;

trait Crontab
{
    /** @var array<int, array<string, string>> */
    private array $commands = [];

    public function addLaravelSchedulerToCrontab(): void
    {
        $this->addCustomCommandToCrontab(
            command: 'artisan schedule:run',
            cronRule: '* * * * * php {{current_path}}/artisan schedule:run > {{current_path}}/storage/logs/schedule.log',
        );
    }

    public function addCustomCommandToCrontab(string $command, string $cronRule): void
    {
        $this->commands[] = [$command => $cronRule];
        $this->processCommands();
    }

    private function processCommands(): void
    {
        if (count($this->commands) === 0) {
            return;
        }

        set('bin/crontab', function () {
            $hostingType = $this->getHostingType();

            return match ($hostingType) {
                HostingType::LinQhost => '/usr/bin/crontab',
                HostingType::Combell => '/usr/local/bin/crontab',
                default => which('crontab')
            };
        });

        task('crontabs:update', function () {
            $hostingType = $this->getHostingType();

            if ($hostingType === HostingType::Forge) {
                return;
            }

            foreach ($this->commands as $commands) {
                foreach ($commands as $command => $cronRule) {
                    if ($this->commandDoesntExist($command)) {
                        $this->addRuleToCrontab($cronRule, $hostingType);
                    }
                }
            }
        });
    }

    private function commandDoesntExist(string $command): bool
    {
        $exists = 'crontab rule exists';
        $missing = 'crontab rule missing';
        $output = run(sprintf("{{bin/crontab}} -l | grep -q '%s' && echo '%s' || echo '%s'", $command, $exists, $missing));

        return str_contains($output, $missing);
    }

    private function addRuleToCrontab(string $command, HostingType $hostingType): void
    {
        if ($hostingType === HostingType::Combell) {
            $tempCrontab = run('mktemp');

            run(sprintf('crontab -l > %s 2>/dev/null || true', $tempCrontab));

            $escapedCronJob = escapeshellarg($command);

            run(sprintf('echo %s >> %s', $escapedCronJob, $tempCrontab));
            run('echo "$(cat '.$tempCrontab.')" > /etc/crontab');

            run('rm '.$tempCrontab);
        }

        if ($hostingType === HostingType::LinQhost) {
            run('{{bin/crontab}} -l | { cat; echo "'.$command.'"; } | {{bin/crontab}} -');
        }

        info('Crontab rule added');
    }

    private function getHostingType(): HostingType
    {
        return HostingType::tryFrom(currentHost()->get('labels')['hostingType']);
    }
}
