<?php

namespace Concept7\Deployer\Connections;

use Illuminate\Support\Str;
use Laravel\Forge\Forge as ForgeForge;
use Laravel\Forge\Resources\Database;
use Laravel\Forge\Resources\DatabaseUser;
use Laravel\Forge\Resources\Site;

class Forge
{
    private ForgeForge $forge;

    public function __construct(string $apiKey, private int $serverId)
    {
        $this->forge = new ForgeForge($apiKey);
    }

    public function createSite(string $siteUrl, string $phpVersion = 'php83'): void
    {
        $site = $this->findSite($siteUrl);

        if (! is_null($site)) {
            return;
        }

        $site = $this->forge->createSite($this->serverId, [
            'domain' => $siteUrl,
            'project_type' => 'php',
            'directory' => '/current/public',
            'php_version' => $phpVersion,
        ]);

        $name = $this->parseDatabaseNameFromSiteUrl($siteUrl);
        $this->forge->createDatabase($this->serverId, [
            'name' => $name,
            'user' => $name,
            'password' => $name,
        ]);

        $this->forge->obtainLetsEncryptCertificate(
            $this->serverId,
            $site->id,
            [
                'domains' => [$siteUrl],
            ],
            false
        );
    }

    public function deleteSite(string $siteUrl): void
    {
        $site = $this->findSite($siteUrl);

        if (! is_null($site)) {
            $this->forge->deleteSite($this->serverId, $site->id);
        }

        $database = $this->findDatabase($this->parseDatabaseNameFromSiteUrl($siteUrl));
        if (! is_null($database)) {
            $this->forge->deleteDatabase($this->serverId, $database->id);
        }

        $user = $this->findUser($this->parseDatabaseNameFromSiteUrl($siteUrl));
        if (! is_null($user)) {
            $this->forge->deleteDatabaseUser($this->serverId, $user->id);
        }
    }

    private function findSite(string $siteUrl): ?Site
    {
        return collect($this->forge->sites($this->serverId))
            ->first(static fn (Site $site) => $site->name === $siteUrl);
    }

    private function findDatabase(string $databaseName): ?Database
    {
        return collect($this->forge->databases($this->serverId))
            ->first(static fn (Database $database) => $database->name === $databaseName);
    }

    private function findUser(string $username): ?DatabaseUser
    {
        return collect($this->forge->databaseUsers($this->serverId))
            ->first(static fn (DatabaseUser $databaseUser) => $databaseUser->name === $username);
    }

    private function parseDatabaseNameFromSiteUrl(string $siteUrl): string
    {
        return Str::of($siteUrl)->before('.')->replace('-', '_')->__toString();
    }
}
