<?php

namespace Concept7\Deployer\Recipe;

use Concept7\Deployer\Traits\PhpFpmReload;

use function Deployer\add;
use function Deployer\after;
use function Deployer\currentHost;
use function Deployer\desc;
use function Deployer\host;
use function Deployer\run;
use function Deployer\set;
use function Deployer\task;
use function Deployer\upload;
use function Deployer\which;

abstract class Base
{
    use PhpFpmReload;

    protected float $phpVersion;

    protected string $repository;

    protected $deploySteps = ['deploy'];

    protected $deployerRecipes = [
        'recipe/common.php',
        'contrib/rsync.php',
        'contrib/cachetool.php',
        'contrib/php-fpm.php',
    ];

    protected $additionalDeployerRecipes = [];

    protected array $environments = [];

    protected string $rootDir;

    public function __construct()
    {
        foreach ($this->deployerRecipes as $deployerRecipe) {
            require_once $deployerRecipe;
        }

        foreach ($this->additionalDeployerRecipes as $additionalDeployerRecipe) {
            require_once $additionalDeployerRecipe;
        }

        $this->rootDir = $this->rootDirectory();
    }

    public function setPhpVersion(float $version)
    {
        $this->phpVersion = $version;

        return $this;
    }

    public function setRepository(string $repository)
    {
        $this->repository = $repository;

        set('repository', $repository);

        return $this;
    }

    public function setApplication(string $name)
    {
        set('application', $name);

        return $this;
    }

    public function setEnvironments(array $environments)
    {
        $this->environments = $environments;

        return $this;
    }

    public function rootDirectory()
    {
        // Change the second parameter to suit your needs
        return dirname(__FILE__, 6);
    }

    public function run(): void
    {
        $this->setupEnvironments();
        $this->setupDotEnv();
        $this->setupDefaults();
        $this->setupGit();
        $this->setupPhp();
        $this->setupComposer();
        $this->setupCopyFrontendAssets();
        $this->setupCacheTool();
        $this->setupPhpFpmReload();
        $this->setupDeploySteps();
        $this->loadExtraDeployerSteps();
    }

    private function setupEnvironments(): void
    {
        foreach ($this->environments as $environment => $data) {
            host($data['host'])
                ->set('labels', ['env' => $environment])
                ->set('hostname', $data['hostname'])
                ->set('port', $data['port'])
                ->set('branch', $data['branch'])
                ->set('deploy_path', $data['deploy_path'])
                ->set('remote_user', $data['remote_user'])
                ->set('ssh_arguments', ['-q -o SendEnv=NOMOTD']);
        }
    }

    private function setupDotEnv(): void
    {
        $env = getenv('ENV');
        set('shared_files', [
            ...($env ? [] : ['.env']),
        ]);

        if ($env) {
            task(
                'deploy:secrets',
                static function () {
                    upload(getenv('ENV'), '{{release_path}}/.env');
                }
            );
            after('deploy:update_code', 'deploy:secrets');
        }
    }

    private function setupDefaults(): void
    {
        set('writable_mode', 'chmod');
        set('ssh_multiplexing', false);
        set('forward_agent', false);
        set('remote_user', 'forge');
        set('release_name', static function () {
            return (string) run('date +"%Y%m%d%H%M%S"');
        });
        set('keep_releases', 2);
        add('shared_dirs', []);
        set('writable_dirs', []);
    }

    private function setupPhp(): void
    {
        set('bin/php', function () {
            return which(
                (currentHost()->get('labels')['env'] === 'production')
                ? 'php'
                : 'php'.$this->phpVersion
            );
        });
    }

    private function setupComposer(): void
    {
        set('bin/composer', static function () {
            return '{{bin/php}} '.which('composer');
        });
    }

    private function setupGit(): void
    {
        set('bin/git', static function () {
            return which('git');
        });
    }

    private function setupCopyFrontendAssets(): void
    {
        set('rsync_src', $this->rootDir.'/public/build');
        set('rsync_dest', '{{release_path}}/public/build');
    }

    private function setupCacheTool(): void
    {
        set('cachetool_args', '--web=FileGetContents --web-path=./public --web-url=https://{{alias}}');
    }

    private function setupDeploySteps(): void
    {
        desc('Deploys your project');
        task('deploy', $this->deploySteps);

        after('deploy:failed', 'deploy:unlock');
    }

    public function setDeploySteps(array $steps): void
    {
        $this->deploySteps = $steps;
    }

    public function loadExtraDeployerSteps(): void
    {
    }
}
