<?php

namespace Concept7\DeployerExtended\Recipe;

use Concept7\DeployerExtended\Traits\PhpFpmReload;

use function Deployer\add;
use function Deployer\after;
use function Deployer\currentHost;
use function Deployer\desc;
use function Deployer\host;
use function Deployer\run;
use function Deployer\set;
use function Deployer\task;
use function Deployer\upload;
use function Deployer\which;

abstract class Base
{
    use PhpFpmReload;

    protected float $phpVersion;

    protected string $repository;

    protected $deploySteps = ['deploy'];

    protected $deployerRecipes = [
        'recipe/common.php',
        'contrib/rsync.php',
        'contrib/cachetool.php',
        'contrib/php-fpm.php',
    ];

    protected $additionalDeployerRecipes = [];

    protected array $stagingEnvironment = [];

    protected array $acceptanceEnvironment = [];

    protected array $productionEnvironment = [];

    protected string $baseDir;

    public function __construct()
    {
        foreach ($this->deployerRecipes as $recipePath) {
            require_once $recipePath;
        }

        foreach ($this->additionalDeployerRecipes as $recipePath) {
            require_once $recipePath;
        }
    }

    public function setPhpVersion(float $version)
    {
        $this->phpVersion = $version;

        return $this;
    }

    public function setRepository(string $repository)
    {
        $this->repository = $repository;

        set('repository', $repository);

        return $this;
    }

    public function setApplication(string $name)
    {
        set('application', $name);

        return $this;
    }

    public function setStaging(
        string $host,
        string $deploy_path,
        string $hostname,
        string $remote_user,
        int $port = 22,
        string $branch = 'develop'
    ) {
        $this->stagingEnvironment = [
            'host' => $host,
            'hostname' => $hostname,
            'remote_user' => $remote_user,
            'port' => $port,
            'branch' => $branch,
            'deploy_path' => $deploy_path,
        ];

        return $this;
    }

    public function setAcceptance(
        string $host,
        string $deploy_path,
        string $hostname,
        string $remote_user,
        int $port = 22,
        string $branch = 'main'
    ) {
        $this->acceptanceEnvironment = [
            'host' => $host,
            'hostname' => $hostname,
            'remote_user' => $remote_user,
            'port' => $port,
            'branch' => $branch,
            'deploy_path' => $deploy_path,
        ];

        return $this;
    }

    public function setProduction(
        string $host,
        string $deploy_path,
        string $hostname,
        string $remote_user,
        int $port = 22,
        string $branch = 'main'
    ) {
        $this->productionEnvironment = [
            'host' => $host,
            'hostname' => $hostname,
            'remote_user' => $remote_user,
            'port' => $port,
            'branch' => $branch,
            'deploy_path' => $deploy_path,
        ];

        return $this;
    }

    public function setBaseDir(string $dir)
    {
        $this->baseDir = $dir;

        return $this;
    }

    public function run(): void
    {
        $this->setupEnvironments();
        $this->setupDotEnv();
        $this->setupDefaults();
        $this->setupGit();
        $this->setupPhp();
        $this->setupComposer();
        $this->setupCopyFrontendAssets();
        $this->setupCacheTool();
        $this->setupPhpFpmReload();
        $this->setupDeploySteps();
        $this->loadExtraDeployerSteps();
    }

    private function setupEnvironments(): void
    {
        $this->setupStaging();
        $this->setupAcceptance();
        $this->setupProduction();
    }

    private function setupStaging(): void
    {
        if (count($this->stagingEnvironment) === 0) {
            return;
        }

        host($this->stagingEnvironment['host'])
            ->set('labels', ['stage' => 'staging'])
            ->set('hostname', $this->stagingEnvironment['hostname'])
            ->set('port', $this->acceptanceEnvironment['port'])
            ->set('branch', $this->stagingEnvironment['branch'])
            ->set('deploy_path', $this->stagingEnvironment['deploy_path'])
            ->set('remote_user', $this->stagingEnvironment['remote_user']);
    }

    private function setupAcceptance(): void
    {
        if (count($this->acceptanceEnvironment) === 0) {
            return;
        }

        host($this->acceptanceEnvironment['host'])
            ->set('labels', ['stage' => 'acceptance'])
            ->set('hostname', $this->acceptanceEnvironment['hostname'])
            ->set('port', $this->acceptanceEnvironment['port'])
            ->set('branch', $this->acceptanceEnvironment['branch'])
            ->set('deploy_path', $this->acceptanceEnvironment['deploy_path'])
            ->set('remote_user', $this->acceptanceEnvironment['remote_user']);
    }

    private function setupProduction(): void
    {
        if (count($this->productionEnvironment) === 0) {
            return;
        }

        host($this->productionEnvironment['host'])
            ->set('labels', ['stage' => 'production'])
            ->set('hostname', $this->productionEnvironment['hostname'])
            ->set('port', $this->productionEnvironment['port'])
            ->set('ssh_arguments', ['-q -o SendEnv=NOMOTD'])
            ->set('branch', $this->productionEnvironment['branch'])
            ->set('deploy_path', $this->productionEnvironment['deploy_path'])
            ->set('remote_user', $this->productionEnvironment['remote_user'])
            ->set('bin/php', function () {
                return which('php');
            })
            ->set('keep_releases', 5);
    }

    private function setupDotEnv(): void
    {
        $env = getenv('ENV');
        set('shared_files', [
            ...($env ? [] : ['.env']),
        ]);

        if ($env) {
            task(
                'deploy:secrets',
                function () {
                    upload(getenv('ENV'), '{{release_path}}/.env');
                }
            );
            after('deploy:update_code', 'deploy:secrets');
        }
    }

    private function setupDefaults(): void
    {
        set('writable_mode', 'chmod');
        set('ssh_multiplexing', false);
        set('forward_agent', false);
        set('remote_user', 'forge');
        set('release_name', function () {
            return (string) run('date +"%Y%m%d%H%M%S"');
        });
        set('keep_releases', 2);
        add('shared_dirs', []);
        set('writable_dirs', []);
    }

    private function setupPhp(): void
    {
        set('bin/php', function () {
            return which(
                (currentHost()->get('labels')['stage'] === 'production')
                ? 'php'
                : 'php'.$this->phpVersion
            );
        });
    }

    private function setupComposer(): void
    {
        set('bin/composer', function () {
            return '{{bin/php}} '.which('composer');
        });
    }

    private function setupGit(): void
    {
        set('bin/git', function () {
            return which('git');
        });
    }

    private function setupCopyFrontendAssets(): void
    {
        set('rsync_src', $this->baseDir.'/public/build');
        set('rsync_dest', '{{release_path}}/public/build');
    }

    private function setupCacheTool(): void
    {
        set('cachetool_args', '--web=FileGetContents --web-path=./public --web-url=https://{{alias}}');
    }

    private function get($array, $key, $default = null)
    {
        if (array_key_exists($key, $array)) {
            return $array[$key];
        }

        return $default;
    }

    private function setupDeploySteps(): void
    {
        desc('Deploys your project');
        task('deploy', $this->deploySteps);

        after('deploy:failed', 'deploy:unlock');
    }

    public function setDeploySteps(array $steps): void
    {
        $this->deploySteps = $steps;
    }

    public function loadExtraDeployerSteps(): void
    {

    }
}
