<?php

namespace Concept7\Deployer\Recipe;

use Composer\InstalledVersions;
use Concept7\Deployer\Enum\HostingType;
use Concept7\Deployer\Traits\PhpRestart;

use function Deployer\add;
use function Deployer\after;
use function Deployer\currentHost;
use function Deployer\desc;
use function Deployer\host;
use function Deployer\run;
use function Deployer\set;
use function Deployer\task;
use function Deployer\upload;
use function Deployer\which;

abstract class Base
{
    use PhpRestart;

    protected float $phpVersion;

    protected string $repository;

    /** @var array<string> */
    protected array $deploySteps = ['deploy'];

    /** @var array<string> */
    protected array $deployerRecipes = [
        'recipe/common.php',
        'contrib/rsync.php',
        'contrib/cachetool.php',
        'contrib/php-fpm.php',
    ];

    /** @var array<string> */
    protected array $additionalDeployerRecipes = [];

    /** @var array<array<string>> */
    protected array $environments = [];

    protected string $rootDir;

    protected bool $hasDotEnv = false;

    public function __construct()
    {
        foreach ($this->deployerRecipes as $deployerRecipe) {
            require_once $deployerRecipe;
        }

        foreach ($this->additionalDeployerRecipes as $additionalDeployerRecipe) {
            require_once $additionalDeployerRecipe;
        }

        $this->rootDir = $this->rootDirectory();
    }

    public function setPhpVersion(float $version): self
    {
        $this->phpVersion = $version;

        return $this;
    }

    public function setRepository(string $repository): self
    {
        $this->repository = $repository;

        set('repository', $repository);

        return $this;
    }

    public function setApplication(string $name): self
    {
        set('application', $name);

        return $this;
    }

    /** @param  array<array<string>>  $environments */
    public function setEnvironments(array $environments): self
    {
        $this->environments = $environments;

        return $this;
    }

    public function rootDirectory(): string
    {
        // Change the second parameter to suit your needs
        return dirname(__FILE__, 6);
    }

    public function run(): void
    {
        $this->setupEnvironments();
        $this->setupDefaults();
        $this->setupDotEnv();
        $this->setupGit();
        $this->setupPhp();
        $this->setupComposer();
        $this->setupCopyFrontendAssets();
        $this->setupCacheTool();
        $this->phpRestart();
        $this->setupDeploySteps();
        $this->loadExtraDeployerSteps();
    }

    private function setupEnvironments(): void
    {
        foreach ($this->environments as $environment => $data) {
            /** @var HostingType */
            $hosting_type = $data['hosting_type'];

            host($data['host'])
                ->set('labels', [
                    'env' => $environment,
                    'hostingType' => $hosting_type->value,
                ])
                ->set('queue_worker_name', $data['queue_worker_name'])
                ->set('websocket_worker_name', $data['websocket_worker_name'])
                ->set('hostname', $data['hostname'])
                ->set('port', $data['port'])
                ->set('branch', $data['branch'])
                ->set('deploy_path', $data['deploy_path'])
                ->set('remote_user', $data['remote_user'])
                ->set('ssh_arguments', ['-q -o SendEnv=NOMOTD']);
        }
    }

    private function setupDotEnv(): void
    {
        $env = getenv('ENV');

        if ($env) {
            $this->hasDotEnv = true;

            set('shared_files', []);

            task(
                'deploy:secrets',
                static function () {
                    upload(getenv('ENV'), '{{release_path}}/.env');
                }
            );
            after('deploy:update_code', 'deploy:secrets');
        }
    }

    private function setupDefaults(): void
    {
        set('writable_mode', 'chmod');
        set('ssh_multiplexing', false);
        set('forward_agent', false);
        set('remote_user', 'forge');
        set('release_name', static function () {
            return (string) run('date +"%Y%m%d%H%M%S"');
        });
        set('keep_releases', 2);
        add('shared_dirs', []);
        set('writable_dirs', []);
    }

    private function setupPhp(): void
    {
        set('bin/php', function () {
            return which(
                (currentHost()->get('labels')['env'] === 'production')
                ? 'php'
                : 'php'.$this->phpVersion
            );
        });
    }

    private function setupComposer(): void
    {
        set('bin/composer', static function () {
            return '{{bin/php}} '.which('composer');
        });
    }

    private function setupGit(): void
    {
        set('bin/git', static function () {
            return which('git');
        });
    }

    private function setupCopyFrontendAssets(): void
    {
        set('rsync_src', $this->rootDir.'/public/build');
        set('rsync_dest', '{{release_path}}/public/build');
    }

    private function setupCacheTool(): void
    {
        set('cachetool_args', '--web=FileGetContents --web-path=./public --web-url=https://{{alias}}');
    }

    private function setupDeploySteps(): void
    {
        desc('Deploys your project');

        $deploySteps = $this->filterBasedOnInstalledComposerPackages();

        task('deploy', $deploySteps);

        after('deploy:failed', 'deploy:unlock');
    }

    /** @param  array<string>  $steps */
    public function setDeploySteps(array $steps): void
    {
        $this->deploySteps = $steps;
    }

    public function loadExtraDeployerSteps(): void
    {
    }

    /** @return array<string> */
    private function filterBasedOnInstalledComposerPackages(): array
    {
        $filterOn = [
            'laravel/horizon' => 'horizon:restart',
            'beyondcode/laravel-websockets' => 'websockets:restart',
        ];
        $installedComposerPackages = InstalledVersions::getInstalledPackages();

        $deploySteps = $this->deploySteps;
        foreach ($filterOn as $package => $taskName) {
            $deploySteps = array_filter(
                $deploySteps,
                static fn ($step) => $step !== $taskName || ($step === $taskName && in_array($package, $installedComposerPackages))
            );
        }

        return $deploySteps;
    }
}
